import viz
import math
import time
import random

#========================================================================
#========================================================================
# Fine tune the Avatar's heading and expression idling here

EXP_DURATION 	= 1.1	# sec, how long to form a facial expression
HEAD_MOVE_MIN	= 3.0	# sec, minimum head moving time
HEAD_MOVE_MAX	= 9.0	# sec, maximum head moving time
HEAD_PAUSE_MIN	= 0.5	# sec, minimum pause between head movements
HEAD_PAUSE_MAX	= 1.5	# sec, maximum pause between head movements

# Random movement periods in Yaw, Pitch, and Roll axes (sec)
YPERIODS = [.5, 0.33, 0.12]
PPERIODS = [.5, 0.33, 0.12]
RPERIODS = [0.12, .05]
# Scale factors for how far to move head (deg)
YSCALE	 = 0.8
PSCALE	 = 1.0
RSCALE   = 1.5

# Adjust frequency of facial expressions.  Number after '*' is odds out of
# 100 that the expression will be played.  The sum of all expressions odds
# must not exceed 100.
#	5, 6	eye brown
#	7		smile
#	9		wide-eye
#	11		squint
#	2		frown
TARG_EXP = [[5,6]] *50 + \
		   [[5]]   *5  + \
		   [[7]]   *10 + \
		   [[9]]   *6  + \
		   [[11]]  *6 

TARG_EXP = [[7,8]] *30 + \
		   [[0]]   *5  + \
		   [[2]]   *10 + \
		   [[11]]   *6  + \
		   [[14]]  *30 


#========================================================================
#========================================================================


# Timers
HEAD_IDLE			= 1
PLAY_EXP			= 2
RESUME_HEAD_IDLE	= 3

class AvatarExpressionIdler(viz.EventClass):

	ALL_EXP  = (100 - len(TARG_EXP)) * [[]] + TARG_EXP
	
	def __init__(self, avatar, face):
		viz.EventClass.__init__(self)
		self.avatar = avatar
		#Get a handle to the Neck bone
		self.neck = self.avatar.getbone('skel_Neck')
		self.neck.lock() #Disable automatic bone animation so that we can manually animate it
		#Lock the head bone so that manual movement of the neck bone also moves the child head bone
		self.avatar.getbone('skel_Head').lock()
		self.face = face
		self.expStartTime = 0.0
		self.expTarget = []
		self.headIdleTime = 0.0
		self.euler = [0.0, 0.0, 0.0]
		self.lastTime = time.time()
		self.deltaTime = 0.0
		self.goHeadIdle = 0
		self.goExpressionIdle = 0
		
		self.callback(viz.TIMER_EVENT, self.mytimer)

	def mytimer(self, num):
		if num == HEAD_IDLE and self.goHeadIdle:
			if self.headIdleTime > time.time():		
			
				t = time.time() - self.deltaTime
				yaw = 0.0
				for p in YPERIODS:
					yaw += math.sin(t*2*math.pi*p)

				pitch = 0.0
				for p in PPERIODS:
					pitch += math.sin(t*2*math.pi*p + 133.234)

				roll = 0.0
				for p in RPERIODS:
					roll += math.sin(t*2*math.pi*p + 958.12)
					
				self.neck.rotate(pitch*PSCALE, yaw*YSCALE, roll*RSCALE)
				self.euler = [yaw, pitch, roll]
				self.starttimer(HEAD_IDLE)

			else:
				self.lastTime = time.time()
				self.starttimer(RESUME_HEAD_IDLE, random.uniform(HEAD_PAUSE_MIN, HEAD_PAUSE_MAX))

		if num == RESUME_HEAD_IDLE:
			self.StartHeadIdle()
		
		if num == PLAY_EXP and self.goExpressionIdle:
			elapsed = time.time() - self.expStartTime
			if elapsed < EXP_DURATION:
				amount = math.sin(elapsed/EXP_DURATION*math.pi)
				for t in self.expTarget:
					self.face.morph(t, amount)
				self.starttimer(PLAY_EXP)
			else:
				self.StartExpressionIdle()

	def StartExpressionIdle(self):
		self.goExpressionIdle = 1
		self.expTarget = random.choice(self.ALL_EXP)
		self.expStartTime = time.time()
		self.starttimer(PLAY_EXP)
		
	def StartHeadIdle(self):
		self.goHeadIdle = 1
		self.headIdleTime = time.time() + random.uniform(HEAD_MOVE_MIN, HEAD_MOVE_MAX)
		self.deltaTime += time.time() - self.lastTime
		self.starttimer(HEAD_IDLE)

	def StopExpressionIdle(self):
		self.goExpressionIdle = 0

	def StopHeadIdle(self):
		self.goHeadIdle = 0
