import viz

#Change Log
#06/03/31: Added support for newline characters in ipc command strings

BUFFER_SIZE = 1000 #Number of commands to keep in buffer

SPECIAL_CHARS = { '\n':r'\n' }

class Recorder(viz.EventClass):
	def __init__(self):
		#Initialize event class
		viz.EventClass.__init__(self)
		
		#Setup callback for exiting and frame
		self.callback(viz.EXIT_EVENT,self.__onexit)
		self.callback(viz.FRAME_EVENT,self.__onframe)
		
		#Recording flag
		self.__recording = False
		
		#Stores current frame number
		self.__frame = 0
		
		#File number
		self.__file = None
		
		#Buffered commands
		self.__buffer = []
		
		#Save reference to original IPC function
		self.__originalIPC = viz._ipcSend
		
	#Define function that will intercept IPC calls
	def __interceptIPC(self,command, p1, p2, mesg, x, y, z, q):
		if self.__recording:
			#Record command
			self.saveCommand(command,p1,p2,mesg,x,y,z,q)
		#Pass command to original function
		return self.__originalIPC(command,p1,p2,mesg,x,y,z,q)
		
	def __writeCommand(self,command):
		#Write command to file
		self.__file.write(','.join([ str(x) for x in command ]))
		self.__file.write('\n')
		
	def __flush(self):
		#Write buffered commands
		for command in self.__buffer:
			self.__writeCommand(command)
			
		#Clear buffer
		del self.__buffer[:]
		
	def __onexit(self):
		#Stop recording on exit
		self.stopRecording()
		
	def __onframe(self):
		
		#Write frame header
		self.saveCommand(-1,self.__frame,0,'',0,0,0,0)
		
		#Increment frame number
		self.__frame += 1
		
		#Flush buffer if exceeded length
		if len(self.__buffer) > BUFFER_SIZE:
			self.__flush()
			
	def saveCommand(self, command, p1, p2, mesg, x, y, z, q):
		#check for special characters		
		for r in SPECIAL_CHARS:
			mesg = mesg.replace( r, SPECIAL_CHARS[r] )
		
		self.__buffer.append((int(command),int(p1),int(p2),mesg,float(x),float(y),float(z),float(q)))
			
	def startRecording(self,filename):
		#Stop any current recording in progress
		self.stopRecording()
		#Open file for writing
		self.__file = open(filename,'w')
		#Turn on recording flag
		self.__recording = True
		#Replace viz ipc function with ours
		viz._ipcSend		= self.__interceptIPC
		viz._ipcSendHold	= self.__interceptIPC
		
	def stopRecording(self):
		if self.__file:
			#Flush output
			self.__flush()
			#Close file
			self.__file = None
			#Turn off recording flag
			self.__recording = False
			#Restore original viz ipc function
			viz._ipcSend		= self.__originalIPC
			viz._ipcSendHold	= self.__originalIPC

	def pause(self):
		self.__recording = False
		
	def resume(self):
		self.__recording = True
		
class Playback(viz.EventClass):
	
	def __init__(self):
		#Initialize event class
		viz.EventClass.__init__(self)
		
		#Setup callback for frame
		self.callback(viz.FRAME_EVENT,self.__onframe)
		
		#Playing flag
		self.__playing = False
		
		#Stores current frame number
		self.__frame = 0
		
		#Buffered commands
		self.__frames = []		
		
	def __applyFrame(self,frame):
		if frame < len(self.__frames):
			for command in self.__frames[frame]:
				viz._ipcSend(*command)
		
	def __onframe(self):
		
		#Play current frame
		self.__applyFrame(self.__frame)
		
		#Increment frame number
		self.__frame += 1
		
	def play(self,filename):
		#Delete previous frame data
		del self.__frames[:]
		#Read in file
		fp = open(filename,'r')
		frame = []
		for line in fp:
			s = line.split(',')
			cmd = int(s[0])
			if cmd == -1:
				self.__frames.append(frame)
				frame = []
			else:
				#check for special characters		
				for r in SPECIAL_CHARS:
					s[3] = s[3].replace( SPECIAL_CHARS[r], r )
				frame.append((int(s[0]),int(s[1]),int(s[2]),s[3],float(s[4]),float(s[5]),float(s[6]),float(s[7])))
		#Reset frame number
		self.__frame = 0