﻿'''
There are four animation paths visualized with spheres of different colors. 
The red spheres mark where these paths intersect. A path will randomly be
selected when the avatar reaches an intersection.
'''
import viz
import vizact
import viztask
import vizinfo
import vizdlg
import vizshape
import random

viz.fov(60)
viz.setMultiSample(4)
viz.go()

vizinfo.InfoPanel()

viz.clearcolor(viz.SLATE)
viz.add('maze.osgb')

import vizcam
cam = vizcam.PivotNavigate(distance=23)
cam.rotateUp(40)

#Add a walking avatar
avatar = viz.add('vcc_male2.cfg')
avatar.state(2)

path_names = ['Blue', 'Green', 'Purple', 'Yellow']

#Colors to visualize the different paths
color = { 'Blue': viz.BLUE, 'Green': viz.GREEN, 'Purple': viz.PURPLE, 'Yellow': viz.YELLOW }

#Create the animation paths
path = {}
for name in path_names:
    path[name] = viz.addAnimationPath()

#Initialize a dictionary of control points for each path
position = {
    'Blue':[ [-4,0,-4], [-4,0,0], [-4,0,5], [0,0,5], [0,0,0], [5,0,0], [5,0,-4] ],
    'Green':[ [5,0,-4], [8,0,-4], [11,0,-4], [11,0,-9], [8,0,-9] ],
    'Purple':[ [5,0,-4], [1.5,0,-4], [-1.5,0,-4], [-4,0,-4] ],
    'Yellow':[ [-4,0,-4], [-4,0,-9], [-1.5,0,-9], [1.5,0,-9], [5,0,-9] ],
}

#Add and visualize control points
def addControlPointToPath(path, position, color):
    for x,pos in enumerate(position):
        b = vizshape.addSphere(radius=0.2,pos=pos,alpha=0.7,color=color)
        path.addControlPoint(x+1,pos=pos)
        
        #set color of junctions to red
        if b.getPosition() in [[-4,0,-4],[5,0,-4]]:
            b.color(viz.RED)

for name in path_names:
    addControlPointToPath(path[name], position[name], color[name])

#Add begin and end events to paths, enable automatic rotation and looping
for name in path_names:
    path[name].addEventAtTime('begin_'+name, 0.0)
    path[name].addEventAtEnd('end_'+name)
    path[name].setAutoRotate(viz.ON)
    path[name].setLoopMode(viz.SWING)

#Play blue path and link the avatar
path['Blue'].play()
pathLink = viz.link(path['Blue'], avatar)

#define begin and end event conditions
path_begin = {}
path_end = {}
for name in path_names:
    path_begin[name] = viztask.waitPathEvent(path[name], 'begin_'+name)
    path_end[name] = viztask.waitPathEvent(path[name], 'end_'+name)

#Move the avatar along the various paths
def PathTask():
    
    oldPath = path['Blue']
    newPath = path['Blue']
    
    #paths B and D loop back to beginning. This variable lets
    #us know the path has completed the loop and is not at the beginning
    finishedLoop = False
    
    while True:
        
        # Wait for one of the begin or end events to occur
        # based on the event choose the new path for the avatar
        
        d = yield viztask.waitAny( path_begin.values() + path_end.values() )
        condition = d.condition
        
        #When path A finishes, choose path B or C
        if condition == path_end['Blue']:
            oldPath = path['Blue']
            newPath = random.choice([path['Green'], path['Purple']])
        
        #When path B finishes loop, move to path C
        elif condition == path_begin['Green'] and finishedLoop:
            oldPath = path['Green']
            newPath = path['Purple']
            finishedLoop = False
        
        elif condition == path_end['Green']:
            finishedLoop = True
        
        #when path C finishes, move to path A or D  
        elif condition == path_end['Purple']:
            oldPath = path['Purple']
            newPath = random.choice([path['Blue'], path['Yellow']])
        
        elif condition == path_end['Yellow']:
            finishedLoop = True
        
        #When path D finishes loop, move to path A
        elif condition == path_begin['Yellow'] and finishedLoop:
            oldPath = path['Yellow']
            newPath = path['Blue']
            finishedLoop = False

        #Pause old path and reset to beginning. Play new path and link avatar to it 
        oldPath.pause()
        oldPath.reset()
        newPath.play()
        pathLink = viz.link(newPath, avatar)
        
        #Print the name of the event that was generated
        print d.data.name

viztask.schedule(PathTask())

#Setup path control panel
controlPanel = vizinfo.InfoPanel('Path Speeds',align=viz.ALIGN_LEFT_BOTTOM, icon=False)
controlPanel.addSeparator()

#Adjust the path speed when a slider event occurs
def changeSpeed(pos,name):
    path[name].setSpeed(pos*10)

#Add sliders to control path speeds
slider = {}
for name in path_names:
    row = vizdlg.Panel(layout=vizdlg.LAYOUT_HORZ_BOTTOM,border=False,background=False,margin=0)
    row.addItem(viz.addTexQuad(color=color[name],scale=[15,15,1]))
    slider[name] = row.addItem(viz.addSlider())
    slider[name].set(0.1)
    controlPanel.addItem(row)
    vizact.onslider(slider[name], changeSpeed, name)