########################################################################################
#
# Avatar Lemming demo
#
#	A bunch of avatars falling into the 1523 pit room
#
#	Commands:
#		'r' - Reset tracking
#		'a' - Turn automatically spawning avatars ON/OFF
#		's' - Manually spawn an avatar
#		'm' - Morph all heads to male head
#		'f' - Morph all heads to female head
#		'o' - Morph all heads back to their original state
#
########################################################################################

import viz
from whrandom import randint,uniform
import math
import vizmat

viz.go(viz.PROMPT)

#Number of avatars
NUM_AVATARS = 20

#Min/Max time between spawning avatars
SPAWN_TIME = [0.4,1.0]

#Starting position and orientation of avatars
STARTING_POS = [-5,0,6]
STARTING_ROT = [0,1,0,180]

#Height of the pit
PIT_HEIGHT	= -7.0

#Time to morph from one head to another (seconds)
MORPH_TIME = 2.0

#Timer numbers
SPAWN_TIMER = 0
PIT_TIMER	= 1
BLEND_TIMER	= 2

#The different faces to use
faces = ['allison.vzf','emily.vzf','eric.vzf','gerron.vzf']

#The different textures for the avatar skin
skins = ['male0.jpg','male1.jpg','male2.jpg','male3.jpg','male4.jpg','male5.jpg']

#Fragment program that blends textures
blend = viz.add('multitexblend.fp')
blend.param(0,1)
blend.param(1,0)
blend.val = 0.0

#Texture for male morph
texblend1 = viz.add('kc.jpg')
texblend1.wrap(viz.WRAP_S,viz.REPEAT)
texblend1.wrap(viz.WRAP_T,viz.REPEAT)

#Texture for female morph
texblend2 = viz.add('allison.jpg')
texblend2.wrap(viz.WRAP_S,viz.REPEAT)
texblend2.wrap(viz.WRAP_T,viz.REPEAT)

#0 is male, 1 is female
morphState = 0

#Custom falling action
class FallAction(viz.ActionClass):

	def begin(self,object):
		yaw = object.get(viz.EULER)[0]
		self.vector = viz.Vector([math.sin(yaw*vizmat.DEG_TO_RAD),0,math.cos(yaw*vizmat.DEG_TO_RAD)])
		self.vector.normalize()
		self.vector *= 1
		self.pos = viz.Vector(object.get(viz.POSITION))

	def update(self,elapsed,object):

		if self.finished():
			return

		inc = elapsed
		
		self.vector[1] -= 9.8 * elapsed

		self.pos += self.vector * elapsed

		object.translate(self.pos.get())
		
		if self.pos[1] < PIT_HEIGHT:
			self.end(object)
		
	def end(self,object):
		if not self.finished():
			object.center(0,0.9,0)
			object.translate(uniform(-0.4,0.4),-7.7,uniform(-0.4,0.4))
			object.rotate(randint(-15,15),90,0,'',viz.RELATIVE_LOCAL)
			object.playsound('1523/landed.wav')
			viz.ActionClass.end(self,object) #Call "end" function of base class

#Create a fall action (same for all avatars)
fall = viz.ActionData()
fall.data = []
fall.actionclass = FallAction

room = viz.add('1523/w1523a.wrl')

#Place the pitlid to the bottom
pitlid = room.getchild('pitlid')
pitlid.translate(0,-6.9,0)

#Hide the door
door = room.getchild('door')
door.visible(0)

#Tells whether autospawning is on/off
autospawn = 0

#Falling variables for the pit
falling = 0
fallen = 0
fallheight = 0
fallV = 0

#Region for falling into pit
minx = 0.5
maxx = 1.0
maxz = 1.0

#Waiting list of avatars
waitlist = []

#List of all avatars
lemmings = []

#Create all the avatars and add them to the waiting list
for x in range(NUM_AVATARS):
	if x == 0:
		male = viz.add('male.cfg')
	else:
		male = waitlist[0].copy()
	#Initially hide the avatar
	male.visible(0)
	#Apply a skin to the avatar (use texture cache)
	male.texture(viz.add(skins[x%len(skins)],viz.TEX_2D,1))
	#Add a face to the avatar
	face = male.face(faces[x%len(faces)])
	#Initially hide the face
	face.visible(0)
	#Apply multi texture to face
	face.texture(texblend1,'geom_0',1)
	#Apply the blending fragment program to the face
	face.apply(blend,'geom_0')
	#Add the avatar to the waiting and lemming list
	waitlist.append(male)
	lemmings.append(male)

#This will hold all the possible walking paths
path = []

#The first two walk actions are the same for all paths
walk_1 = waitlist[0].walkto(-5,0,4.5)
walk_2 = waitlist[0].walkto(-3.7,0,3.3)

#Create path 1
walk_3 = waitlist[0].walkto(-3,0,0)
walk_4 = waitlist[0].walkto(-2,0,-3.5)
walk_5 = waitlist[0].walkto(1,0,-3.5)
walk_6 = waitlist[0].walkto(1,0,-1)
path.append(vizact.sequence(walk_1,walk_2,walk_3,walk_4,walk_5,walk_6,fall))

#Create path 2
walk_3 = waitlist[0].walkto(-2,0,0.5)
walk_4 = waitlist[0].walkto(-1,0,0)
path.append(vizact.sequence(walk_1,walk_2,walk_3,walk_4,fall))

#Create path 3
walk_3 = waitlist[0].walkto(-2,0,0)
walk_4 = waitlist[0].walkto(-1,0,-2)
walk_5 = waitlist[0].walkto(1,0,-1)
path.append(vizact.sequence(walk_1,walk_2,walk_3,walk_4,walk_5,fall))

#Create path 4
walk_3 = waitlist[0].walkto(-2,0,3)
walk_4 = waitlist[0].walkto(-1,0,1)
path.append(vizact.sequence(walk_1,walk_2,walk_3,walk_4,fall))

#Create path 5
walk_3 = waitlist[0].walkto(-0.5,0,3)
walk_4 = waitlist[0].walkto(0.5,0,1)
path.append(vizact.sequence(walk_1,walk_2,walk_3,walk_4,fall))

#Create path 6
walk_3 = waitlist[0].walkto(-2,0,3)
walk_4 = waitlist[0].walkto(1,0,3)
walk_5 = waitlist[0].walkto(2,0,0.5)
walk_6 = waitlist[0].walkto(1,0,0)
path.append(vizact.sequence(walk_1,walk_2,walk_3,walk_4,walk_5,walk_6,fall))

if viz.get(viz.TRACKER):
	ppt = viz.add('vizppt.dls')
	#Average over 5 samples
	ppt.command(5,'',5)
	#Don't reset y position
	ppt.command(6,'',1,0,1)
	#Connect to intersense on port 5
	PORT_INTERSENSE = 5
	isense = viz.add('intersense.dls')
	viz.tracker()
	#Set eyeheight to zero since we are getting it from the tracker
	viz.eyeheight(0)

def StartAvatar(p):
	
	#If waiting list is empty then return
	if len(waitlist) == 0:
		return
		
	#Clear the avatars actions
	waitlist[0].clear(viz.ACTION)
	waitlist[0].clear(viz.CURRENT_ACTION)
	#Make the avatar visible
	waitlist[0].visible(1)
	#Make the avatar head visible
	waitlist[0]._face_.visible(1)
	#Translate the avatar to the starting position
	waitlist[0].translate(STARTING_POS)
	#Rotate the avatar to the starting rotation
	waitlist[0].rotate(STARTING_ROT)
	#Reset the avatars center
	waitlist[0].center(0,0,0)
	
	#Get the list of actions for the avatars walk path
	waitlist[0].act(path[p])
		
	#Delete the avatar from the waiting list
	del waitlist[0]

def GetSpawnTime():
	#Return a random number between the min and max spawn time
	return uniform(SPAWN_TIME[0],SPAWN_TIME[1])
	
def mytimer(num):
	if num == PIT_TIMER:
		global falling,fallen,fallV,fallheight
		
		if not falling and not fallen:
			#Check if viewpoint is inside fall region
			pos = viz.get(viz.HEAD_POS)
			x = pos[0]
			z = pos[2]
			if minx < x < maxx and -maxz < z < maxz:
				falling = 1
				fallV = 0
				fallheight = 0
			elif -maxx < x < -minx and -maxz < z < maxz:
				falling = 1
				fallV = 0
				fallheight = 0
				
		if falling:
			#Increment fall velocity (gravity * elapsed time)
			fallV -= 9.8 * viz.elapsed()
			#Increment head position (velocity * elapsed time)
			viz.translate(viz.HEAD_POS,0,fallV*viz.elapsed(),0)
			#Increment fall height
			fallheight += fallV*viz.elapsed()
			#Check if fall height is below pit height
			if fallheight < PIT_HEIGHT:
				#Clear falling flags
				fallen = 1
				falling = 0
				#Play landing sound
				viz.playsound('1523/landed.wav')
			
			
		elif fallen:
			#Check if viewpoint is outside pit
			pos = viz.get(viz.HEAD_POS)
			x = pos[0]
			z = pos[2]
			if not (-maxx < x < maxx and -maxz < z < maxz):
				fallen = 0
				#Reset the head position to floor level
				viz.translate(viz.HEAD_POS,0,-fallheight,0)
				
	elif num == SPAWN_TIMER:
		#Spawn an avatar and assign it a random path
		StartAvatar(randint(0,5))
		#Start a timer to spawn another avatar
		viz.starttimer(SPAWN_TIMER,GetSpawnTime())
		
	elif num == BLEND_TIMER:
		
		#Increment blend value based on elapsed time
		blend.val += viz.elapsed() / MORPH_TIME
		
		#Check if done blending
		if blend.val > 1.0:
			blend.val = 1.0
			viz.killtimer(BLEND_TIMER)
			
		#Set blend value depending on blend direction
		if blend.dir:
			blend.param(0,1.0 - blend.val)
			blend.param(1,blend.val)
		else:
			blend.param(0,blend.val)
			blend.param(1,1.0 - blend.val)

def mykey(key):
	global autospawn,morphState
	if key == 'a':
		autospawn = not autospawn
		if autospawn:
			viz.starttimer(SPAWN_TIMER,GetSpawnTime())
		else:
			viz.killtimer(SPAWN_TIMER)
	elif key == 's':
		StartAvatar(randint(0,5))
	elif key == 'm':
		#Set morph state to male
		morphState = 0
		#Reset fragment program blend parameters
		blend.param(0,1.0)
		blend.param(1,0.0)
		blend.val = 0.0
		blend.dir = 1
		for male in lemmings:
			#Clear the morphs from the head
			male._face_.clear()
			#Begin built-in morph animation
			male._face_.morph(0,1,MORPH_TIME)
			#Apply multi texture to face
			male._face_.texture(texblend1,'geom_0',1)
		#Start the texture blending timer
		viz.killtimer(BLEND_TIMER)
		viz.starttimer(BLEND_TIMER,0.01,viz.FOREVER)
	elif key == 'f':
		#Set morph state to female
		morphState = 1
		#Reset fragment program blend parameters
		blend.param(0,1.0)
		blend.param(1,0.0)
		blend.val = 0.0
		blend.dir = 1
		for male in lemmings:
			#Clear the morphs from the head
			male._face_.clear()
			#Begin built-in morph animation
			male._face_.morph(1,1,MORPH_TIME)
			#Apply multi texture to face
			male._face_.texture(texblend2,'geom_0',1)
		#Start the texture blending timer
		viz.killtimer(BLEND_TIMER)
		viz.starttimer(BLEND_TIMER,0.01,viz.FOREVER)
	elif key == 'o':
		#Reset fragment program blend parameters
		blend.param(0,0.0)
		blend.param(1,1.0)
		blend.val = 0.0
		blend.dir = 0
		for male in lemmings:
			#Clear the morphs from the head
			male._face_.clear()
			#Set the current morph state to 100%
			male._face_.morph(morphState,1)
			#Begin built-in morph animation
			male._face_.morph(morphState,0,MORPH_TIME)
		#Start the texture blending timer
		viz.killtimer(BLEND_TIMER)
		viz.starttimer(BLEND_TIMER,0.01,viz.FOREVER)
	elif key == 'r':
		ppt.reset()
		isense.reset()
		viz.reset(viz.HEAD_POS)
		viz.translate(viz.HEAD_POS,0,0,-2)

def finishedActions(avatar):
	if isinstance(avatar,viz.VizAvatar):
		waitlist.append(avatar)

viz.callback(viz.ACTION_EMPTY_EVENT,finishedActions)
viz.callback(viz.KEYBOARD_EVENT,mykey)
viz.callback(viz.TIMER_EVENT,mytimer)

viz.starttimer(PIT_TIMER,0.01,viz.FOREVER)