
from xmath import spaceToScreenCoords

from xmath import Vec2, Vec3, clamp
import math
import viz
import vizact
import vizjoy

def near(n, val, e = 0.05):
    """
    True iff <n> is near <val>.
    Can be used to work around floating point comparison.
    """
    return val - e < n < val + e

class EventManager:
    """Processes messages to control the world (such as viewpoint parameters and animations)."""
    
    def __init__(self, user):
        self.user = user
        
        JoystickManager(self)
        KeyboardManager(self)
        
        self._trans = Vec2()
        self._rot = Vec2()
        
        # keep viewpoint position updated
        def updatePos():
            """Update the viewpoint position according to the joystick's state."""

            e = 0.05
            if self._trans.length() > e:
                angle, length = self._trans.polarCoords(vizAngle=True)
                a = (angle + self.user.getEuler()[0]) * math.pi / 180.0
                pos = Vec3(self.user.getPosition())
                
                # calculate translation over ground plane (y == 0)
                elapsed = viz.tick() - self._posTime
                trans = Vec3(math.sin(a), 0.0, math.cos(a)) * elapsed * User.WALKING_PACE * length
                self._posTime = viz.tick()
                
                self.user.move(pos + trans)
                
        vizact.ontimer(0, updatePos)

        # keep viewpoint orientation updated
        def updateRot():
            if self._rot.length() > 0:
                x, y = self._rot
                ry, rx, _ = self.user.getEuler()
                
                # derive rotation angle
                degrees = (viz.tick() - self._rotTime) * self.user.ROTATION_SPEED
                xAngle = degrees * x
                yAngle = degrees * y
                self._rotTime = viz.tick()
                
                rotY = ry + xAngle
                rotX = clamp(rx - yAngle, -89.0, 89.0)
                self.user.turn([rotY, rotX, 0.0])
        vizact.ontimer(0, updateRot)
        
    def message(self, event='', *args):
        """Process event messages."""
        
#        print 'Event received:', event, args
        messageTable = {#'joy button a'  : self._openTerraceDoor,
                        #'joy button b'  : self._closeTerraceDoor,
                        #'joy button x'  : lambda: self._nm.send('network test'),
                        #'joy button y'  : lambda: self.world.male.state(11),
                        #'joy button rb'  : lambda: control.globalControlInstance.resetAvatarNumbers(),
                        #'joy button lb'  : lambda: control.globalControlInstance.divideAvatarNumbers(),
                        'joy hat'              : lambda * args: self._viewRot(*args),
                        'joy stick'            : lambda * args: self._viewTrans(*args),
                        'gui close'            : lambda * args: viz.quit(),
                        'gui reset tracker'    : lambda * args: self.world.resetTracker(),
                        'key w'                : lambda * args: self._move(*args),
                        'key s'                : lambda * args: self._move(*args),
                        'key a'                : lambda * args: self._move(*args),
                        'key d'                : lambda * args: self._move(*args),
                        'key q'                : lambda * args: self._look(*args),
                        'key e'                : lambda * args: self._look(*args),
                        'key z'                : lambda * args: self._look(*args),
                        'key c'                : lambda * args: self._look(*args),
                        }
        if messageTable.has_key(event):
            messageTable[event](*args)
        else:
            print 'Event "' + event + '" not in message table.'
    
    def _move(self, x, y, z):

        self._viewTrans(z,x)
        
    
    def _look(self, yaw, pitch, roll):
        self._viewRot(yaw,pitch)
    
    def _viewTrans(self, x, y):
        self._trans = Vec2(x, y)
        self._posTime = viz.tick()
        
    def _viewRot(self, x, y):
        self._rot = Vec2(x, y)
        self._rotTime = viz.tick()

class DummyUser(viz.VizGroup):
    """Represents the user who is viewing and navigating the scene."""

    WALKING_PACE = 1.5#3.0 # walking at full speed, meters / second
    ROTATION_SPEED = 120.0 # degrees / second
    FOV = 50.0 # field of view
        
    def __init__(self):
        node = viz.addGroup()
        viz.VizGroup.__init__(self, node.id)
        self.eyeheight(1.67)
                
        self._dummyPar = viz.addGroup()
        self._dummy = viz.addGroup(parent=self._dummyPar)
        
    def sees(self, obj, maxDist=2.5):
        self._dummyPar.setMatrix(self.getMatrix())
        self._dummy.setPosition(obj.getPosition(viz.ABS_GLOBAL), viz.ABS_GLOBAL)
        x, y, z = spaceToScreenCoords(self._dummy.getPosition(), fov=User.FOV)
        return 0.01 < z < maxDist and - 1.0 < x < 1.0 and - 1.0 < y < 1.0

    def eyeheight(self, height, immidiate=viz.ON):
        self.gEyeHeight = height
        viz.MainView.eyeheight(height, immidiate)

    def move(self, newPos):
        """Check collision and move to suggested or corrected (in case of collision) position."""
        
        funcs = [
                ]
        
        pos = newPos        
        for func in funcs:
            pos = func(pos)            
        
        self.setPosition(pos)
    
    def turn(self, newEuler):
        
        self.setEuler(newEuler)

class User(viz.VizView):
    """Represents the user who is viewing and navigating the scene."""

    WALKING_PACE = 1.5#3.0 # walking at full speed, meters / second
    ROTATION_SPEED = 120.0 # degrees / second
    FOV = 50.0 # field of view
        
    def __init__(self):
        viz.VizView.__init__(self, viz.MainView)

        self.setPosition(1, 0, -0.5)
        self.setEuler(180, 0, 0)
#        self.setAxisAngle(0, 1, 0, 180)
        
        self.eyeheight(1.67)
        self.size = 0.2
        
        self.collision(viz.OFF)
        self.collisionbuffer(0.25)
        self.stepsize(0.21) # fix for not stepping over avatars and other stuff
#        self.gravity(100)

        self.squares = [] # to be filled with bounding squares when tables are created
        
        self._dummyPar = SceneNode()
        self._dummy = SceneNode(parent=self._dummyPar)
        
    def sees(self, obj, maxDist=2.5):
        self._dummyPar.setMatrix(self.getMatrix())
        self._dummy.setPosition(obj.getPosition(viz.ABS_GLOBAL), viz.ABS_GLOBAL)
        x, y, z = spaceToScreenCoords(self._dummy.getPosition(), fov=User.FOV)
        return 0.01 < z < maxDist and - 1.0 < x < 1.0 and - 1.0 < y < 1.0

    def eyeheight(self, height, immidiate=viz.ON):
        self.gEyeHeight = height
        viz.VizView.eyeheight(self, height, immidiate)

    def move(self, newPos):
        """Check collision and move to suggested or corrected (in case of collision) position."""
        
        funcs = [
                ]
        
        pos = newPos        
        for func in funcs:
            pos = func(pos)            
            
        self.setPosition(pos)
    
    def turn(self, newEuler):
        
        self.setPosition(newEuler)

class KeyboardManager:
    
    def __init__(self, eventManager):
        
        self.eventManager = eventManager
        
        moveSpeed = 1
        lookSpeed = 5
        
        vizact.onkeydown('w', self.eventManager.message, 'key w', 1, 0,0)
        vizact.onkeydown('s', self.eventManager.message, 'key s', -1, 0,0)
        vizact.onkeydown('a', self.eventManager.message, 'key a', 0,0,-1)
        vizact.onkeydown('d', self.eventManager.message, 'key d', 0,0,1)
        
        vizact.onkeyup('w', self.eventManager.message, 'key w', 0, 0,0)
        vizact.onkeyup('s', self.eventManager.message, 'key s', 0, 0,0)
        vizact.onkeyup('a', self.eventManager.message, 'key a', 0,0,0)
        vizact.onkeyup('d', self.eventManager.message, 'key d', 0,0,0)
        
        vizact.onkeydown('q', self.eventManager.message, 'key q', -1,0,0)
        vizact.onkeydown('e', self.eventManager.message, 'key e', 1,0,0)
        vizact.onkeydown('z', self.eventManager.message, 'key z', 0,1,0)
        vizact.onkeydown('c', self.eventManager.message, 'key c', 0,-1,0)
        
        vizact.onkeyup('q', self.eventManager.message, 'key q', 0,0,0)
        vizact.onkeyup('e', self.eventManager.message, 'key e', 0,0,0)
        vizact.onkeyup('z', self.eventManager.message, 'key z', 0,0,0)
        vizact.onkeyup('c', self.eventManager.message, 'key c', 0,0,0)
    
class JoystickManager:
    def __init__(self, eventManager):

        self.eventManager = eventManager

        # add joystick
        self.joy = vizjoy.add()

        viz.callback(vizjoy.BUTTONDOWN_EVENT, self.joydown)
        viz.callback(vizjoy.BUTTONUP_EVENT, self.joyup)
        viz.callback(vizjoy.HAT_EVENT, self._joyHat)
        viz.callback(vizjoy.MOVE_EVENT, self.joymove)

    def joymove(self, e):
        """Joystick position has changed, update camera position.

        e._startPos	 - new position value
        e.oldPos - old position value
        """

        self.eventManager.message('joy stick', e.pos[0], -e.pos[1])

    def _joyHat(self, e):
        """The joystick hat (arrow button pad) changed, rotate the viewpoint.
        
        e.hat		- new hat value
        e.oldHat	- old hat value
        """
        
        x, y = 0, 0
        if near(e.hat, 0.0):
            y = 1
        elif near(e.hat, 90.0):
            x = 1
        elif near(e.hat, 180.0):
            y = -1
        elif near(e.hat, 270.0):
            x = -1
        
        self.eventManager.message('joy hat', x, y)

    def joydown(self, e):
        """A joystick button is down.
        
        e.button: button number that is down
        """
        
        def m(msg):
            self.eventManager.message('joy button ' + msg)
        {1 : lambda: m('a'),
         2 : lambda: m('b'),
         3 : lambda: m('x'),
         4 : lambda: m('y'),
         5 : lambda: m('lb'),
         6 : lambda: m('rb'),
         7 : lambda: m('back'),
         8 : lambda: m('start'),
         9 : lambda: m('lpush'),
         10: lambda: m('rpush'),
         }[e.button]()

    def joyup(self, e):
        """A joystick button is up.
        
        e.button	- button number that is up
        """
        pass


