"""
VizAction implementations.
"""
import math
import time
import random

from xmath import clamp, lerp
import viz

__author__ = "huib"
__date__ = "$2-dec-2010 19:26:19$"


class Action(viz.ActionClass):
    """Generic case."""

    def begin(self, object):
        pass

    def update(self, elapsed, object):
        pass

    def pause(self, object):
        print "Don't pause the plane control."
        #viz.ActionClass.pause(self, object)

    def end(self, object):
        viz.ActionClass.end(self, object)

    @classmethod
    def getAction(cls, * args):
        """Generic getAction."""
        action = viz.ActionData()
        action.actionclass = cls
        action.data = list(args)
        return action
        


class PlaneBank(Action):
    """
    Plane bank control action for manual control of banking.
    Use setNewBankTarget to change the plane (object) bank
    The direction of the plane (object) will automatically change with the bank
    Maximum recommended bank for a plane is 30 degrees
    Use cleanEnd to exit this action
    The plane will return back to the bank level it had when the action was initiated
    Use getAction() to generate the action object useable by Vizard
    """

    def begin(self, plane):

        self._actiondata_.setNewBankTarget = self.setNewBankTarget
        self._actiondata_.cleanEnd = self.cleanEnd

        self.maximumBankAngle = plane.maxBankAngle
        self.levelBank = plane.getEuler()[2]
        self.currentTargetBank = self.levelBank
        self.currentBank = self.levelBank
        self.lastBank = self.currentBank
        self.lastChange = time.time()
#        self.lastAileronLevel = plane.aileronLevel / plane.maxAileronLevel
        self.plane = plane
        self.doEnd = False

    def setNewBankTarget(self, bankTarget):
        """
        setNewBankTarget sets a new bank target to be reached by the airplane (object)
            recommended maximum bank angle is 30 degrees
        bankTarget = bank angle in degrees
        angle < 0 will produce a left turn, angle > 0 will produce a right turn

        Creates a Vizard usable action for banking control (No forward movement)
        set planeSpeed in Knots per hour
        set banking speed in degrees per second
        """

        bankTarget = -1* clamp(bankTarget, -self.maximumBankAngle, self.maximumBankAngle)
        if self.currentBank != bankTarget:  #Only change the target bank angle if it is different
            self.currentTargetBank = bankTarget
            self.lastBank = self.currentBank
            self.lastChange = time.time()
#            self.lastAileronLevel = self.plane.aileronLevel / self.plane.maxAileronLevel

    def cleanEnd(self):
        """
        Cleanly end the action returning the plane to it original bank angle
        """

        self.setNewBankTarget(self.levelBank)
        self.doEnd = True

    def update(self, elapsed, plane):
        """
        Computes plane bank on certain time.

        See U{http://www.wannamaker.org/aviation/cfi/turns/}.
        """

        #bank phase
        timeDone = time.time() - self.lastChange + elapsed #Time expired after the last set bank angle
        duration = abs((self.currentTargetBank - self.lastBank) / plane.bankingSpeed) #Total duration of the bank angle change
        currentBank = 0.0
        e = 0.001
        if duration != 0: #This only happends if the currentTargetBank equals the lastBank position
            percentageDone = timeDone / duration
            currentBank = lerp(clamp(percentageDone, 0.0, 1.0), self.lastBank, self.currentTargetBank, func='siso') #calculate the bank angle

        if not -e < currentBank < e:
            K = 0.088537 #constant value see website
            radius = (plane.airSpeed ** 2) / math.tan(math.radians(currentBank)) * K #Circle radius of the turn

            K = 0.000009467 #constant value see website
            turnRate = math.sqrt(math.tan(math.radians(currentBank)) / (radius * K)) #turning rate in degrees per second
            if currentBank > 0:
                turnRate *= -1.0
            yaw, pitch, _ = plane.getEuler()
           
            plane.setEuler([yaw + turnRate * elapsed, 0, self.levelBank - currentBank]) #change direction of the airplane
            self.currentBank = currentBank
            
            # Set ailerons to match
#            if (time.time() + elapsed) - self.lastChange < duration:
#                achieveLevel = -clamp(((self.lastBank - self.currentTargetBank) / float(plane.maxBankAngle)), -1.0, 1.0)
#                halfTime = duration / 2.0
#                currentLevel = lerp((time.time() - self.lastChange + elapsed) / halfTime, self.lastAileronLevel, achieveLevel, func='fiso')
#                print self.lastBank - self.currentTargetBank, currentLevel            
#                                
#                if (time.time() + elapsed) - self.lastChange > halfTime:
#                    currentLevel = lerp((time.time() - self.lastChange + elapsed - halfTime) / halfTime, achieveLevel, 0)
#                    
#                plane.aileronsLevel = currentLevel

        if self.doEnd:
            if -e < currentBank < e:
                self.end(plane)

    def end(self, object):
        yaw, pitch, _ = object.getEuler()
        object.setEuler(yaw, pitch, self.levelBank) #put the plane back to starting level
        viz.ActionClass.end(self, object)
    

class PlaneForward(Action):
    """
    Forward movement of the airplane.
    Forward is the direction the planes nose is turned to.

    Speed: plane speed in knots per hour (influences turning rate when banking)
    """

    def update(self, elapsed, plane):
        _, height, _ = plane.getPosition(viz.ABS_GLOBAL)
        plane.setPosition(0, 0, (elapsed * plane.airSpeed * -0.514444444444444),viz.REL_LOCAL)
        x, _, z = plane.getPosition(viz.ABS_GLOBAL)
        plane.setPosition(x,height,z,viz.ABS_GLOBAL)
