#############################################################
# VRUT Python API
#
#------------------------------------------------------------
# Changes:
#
# 11/7/98 ACB
# Switch to using system environment variable for WIN_VRUT_PATH
#
#
#############################################################


import os
import sys
import types
import string
import time

import socket
import struct
import math
import threading


import vizmat

try:
	import ipc
	import vizinput
	# The following path is now based on a system-wide environment
	# variable, _VRUT_PATH, so do not changed this path value manually.
	_WIN_VRUT_PATH = os.environ['VIZARD_PATH']

except:
	print '**ERROR: Vizard has not been installed correctly!'
	import win32ui
	win32ui.MessageBox('Vizard has not been installed correctly!', 'Error', 0)
	sys.exit()



# BUILD VERSION PATHS:
_BIN_PRIMARY = _WIN_VRUT_PATH + '\\bin\\winviz.exe'
_BIN_BETA    = _WIN_VRUT_PATH + '\\bin\\winviz_beta.exe'




if not os.access(_BIN_PRIMARY, os.F_OK) or not os.access(_BIN_BETA, os.F_OK):
	print '**ERROR: Vizard binaries not installed correctly!'
	import win32ui
	win32ui.MessageBox('Vizard binaries not installed correctly!', 'Error', 0)
	sys.exit()   



# OK, things look good, so let's announce VRUT up and running.
vrut_version = '1.08'
print 'Vizard ', vrut_version


_ipcSend = ipc.send
_ipcSendHold = ipc.sendhold
_where = 'X'


def _dummySend(command, p1, p2, mesg, x, y, z, q):
	return

def _dummySendHold(command, p1, p2, mesg, x, y, z, q):
	return -1


################################################################################################
################################################################################################
#									Constant Declarations									   #
################################################################################################
################################################################################################


#************************************************************
# MAIN IPC COMMANDS - these should be copied from vrut.h
# When adding a new VRUT command, give it the SAME name found
# in vrut.h for VRE and the SAME ID#.  Add it to the bottom
# of the list.  
#************************************************************
_VRUT_INITDISPLAY	= 1
_VRUT_STARTTRACKING	= 2
_VRUT_PUSHLAYER		= 3
_VRUT_POPLAYER		= 4
_VRUT_FOG		= 5
_VRUT_CLEARCOLOR		= 6
_VRUT_GL_ENABLE		= 7
_VRUT_GL_DISABLE		= 8
_VRUT_EXECPYTHON		= 9	# run command from VRE 
_VRUT_REGISTERCALLBACK	= 10	# register VRE callback
_VRUT_STARTTIMER		= 11
_VRUT_OPENLAYER		= 12
_VRUT_CLOSELAYER		= 13
_VRUT_BEGINLIST		= 14
_VRUT_ENDLIST		= 15
_VRUT_VERTEX3D		= 16
_VRUT_VERTEXCOLOR	= 17
_VRUT_CLOSETRACKER	= 18
_VRUT_TRANSLATE		= 19
_VRUT_ROTATE		= 20
_VRUT_SCALE		= 21
_VRUT_POINTSIZE		= 22
_VRUT_SCREEN		= 23
_VRUT_QUIT		= 24
_VRUT_CLEARALL		= 25
_VRUT_STARTHOTSPOT	= 26
_VRUT_SELECT		= 27
_VRUT_DATAFILTER		= 28
_VRUT_PLAYBACK		= 29
_VRUT_RECORD		= 30
_VRUT_SCROLLLIST		= 31
_VRUT_OFFSETVIEWT	= 32
_VRUT_OFFSETVIEWR	= 33
_VRUT_RESETLAYER		= 34
_VRUT_RESETVIEWT		= 35
_VRUT_RESETVIEWR		= 36
_VRUT_CLAMP		= 37
_VRUT_TRIGGERDATA	= 38
_VRUT_SETFOV		= 39
_VRUT_PUSHTEXTURE	= 40
_VRUT_BINDTEXTURE	= 41
_VRUT_CURTAINEYE		= 42
_VRUT_SETIPD		= 43
_VRUT_TOGGLE_LAYER	= 44
_VRUT_LIGHTING		= 45
_VRUT_LIGHTAMBIENT	= 46
_VRUT_LIGHTPOSITION	= 47
_VRUT_LIGHTINTENSITY	= 48
_VRUT_LIGHTCOLOR 	= 49
_VRUT_LIGHTSPREAD 	= 50
_VRUT_LIGHTDIRECTION 	= 51
_VRUT_LIGHTENABLE	= 52
_VRUT_LIGHTDISABLE	= 53
_VRUT_REPOSITION_WINDOW  = 54
_VRUT_TOGGLE_LAYEREYE	= 55
_VRUT_PUSHGROUND		= 56
_VRUT_GROUNDBODY		= 57
_VRUT_ECHOFRAMERATE	= 58
_VRUT_SHADOW		= 59
_VRUT_CLIP		= 60
_VRUT_EYEHEIGHT          = 61
_VRUT_PATH		= 62
_VRUT_DELETEHOTSPOT	= 63
_VRUT_PUSHATTRIBUTES	= 64
_VRUT_POPATTRIBUTES		= 65
_VRUT_SHOWMESSAGE		= 66
_VRUT_MATERIAL           = 67
_VRUT_READPIXEL          = 68
_VRUT_WATCH				= 69  ## Added at MIT since leaving UCSB
_VRUT_MESSAGECOLOR		= 70

_VRUT_LIGHTTEST			= 99
_VRUT_ROTATECHILD		= 100
_VRUT_TRANSLATECHILD		= 101
_VRUT_ADDCHILD			= 102
_VRUT_ROTATECHILDR		= 103
_VRUT_TRANSLATECHILDR	= 104
_VRUT_SWITCHCHILD		= 105
_VRUT_CENTERCHILD		= 106
_VRUT_MOVIE1CHILD		= 107
_VRUT_MOVIE2CHILD		= 108
_VRUT_LINEWIDTH			= 109
_VRUT_PLAYSOUND			= 110
_VRUT_SCALECHILD			= 111
_VRUT_SETAPPEARANCE		= 112
_VRUT_ADDSENSOR			= 113
_VRUT_GETSENSOR			= 114
_VRUT_SETASPECT			= 115
_VRUT_RESETSENSOR		= 116
_VRUT_VERSION			= 117
_VRUT_MOUSEMOTIONOFF		= 118
_VRUT_CURSORVISIBLE		= 119
_VRUT_MOUSESCROLL		= 120
_VRUT_ADD_TEXTURE		= 121
_VRUT_APPLY_TEXTURE		= 122
_VRUT_ATTACHSENSOR		= 123
_VRUT_CENTERSENSOR		= 124
_VRUT_SCALESENSOR		= 125
_VRUT_OVERRIDE			= 126
_VRUT_ALPHACHILD			= 127
_VRUT_COMMANDSENSOR		= 128
_VRUT_ADDMAIL			= 129
_VRUT_SENDMAIL			= 130
_VRUT_MOUSECONSTRAIN		= 131
_VRUT_STATSCHILD			= 132
_VRUT_SETSCENE			= 133
_VRUT_SCREENCAPTURE		= 134
_VRUT_TYPESENSOR			= 135
_VRUT_APPLYMIRROR		= 136
_VRUT_SAVECSB			= 137
_VRUT_FINDCHILD			= 138
_VRUT_GETCHILD			= 139
_VRUT_ROTATECHILDEULER	= 140
_VRUT_ROTATECHILDEULER2	= 141
_VRUT_COLLISIONEYEPOINT  = 142
_VRUT_SCRIPTCHILD		= 143
_VRUT_SELF				= 144
_VRUT_SELFCALLBACK		= 145
_VRUT_SELFSTARTTIMER		= 146
_VRUT_SELFADDMAIL		= 147
_VRUT_SELFCOPY			= 148
_VRUT_SELFCLONE			= 149
_VRUT_CLASSCALLBACK		= 150
_VRUT_CLASSSTARTTIMER	= 151
_VRUT_SOUNDCHILD			= 152
_VRUT_ADD_TEXGEN			= 153
_VRUT_SETTRACKING_TYPE	= 154
_VRUT_SETINITOK			= 155
_VRUT_ADDRENDERER		= 156
_VRUT_RESETRENDERER		= 157
_VRUT_COMMANDRENDERER	= 158
_VRUT_ATTACHRENDERER		= 159
_VRUT_SPLASHSCREEN		= 160
_VRUT_REMOVERENDERER		= 161
_VRUT_SUBWINDOW_POS		= 162
_VRUT_SUBDINDOW_SIZE		= 163
_VRUT_ADD_SUBWINDOW		= 164
_VRUT_SUBVIEWPOSITION	= 165
_VRUT_SUBVIEWROTATEEULER = 166
_VRUT_SUBVIEWROTATE		= 167
_VRUT_ADD_SUBVIEWPOINT	= 168
_VRUT_SUBWINDOW_VIEWPOINT= 169
_VRUT_LIGHTSPOTEXPONENT	= 170
_VRUT_HORZ_ABSOLUTE      = 171
_VRUT_HORZ_RELATIVE      = 172
_VRUT_VERT_ABSOLUTE      = 173
_VRUT_VERT_RELATIVE      = 174
_VRUT_STAYONTOP			= 175
_VRUT_KILLTIMER			= 176
_VRUT_SUBWINDOW_CURTAIN	= 177
_VRUT_EXECPYTHON2		= 178
_VRUT_ADDTEXT			= 179
_VRUT_COLORTEXT			= 180
_VRUT_SETVIEWDIST		= 181
_VRUT_SETMATRIX1			= 182
_VRUT_SETMATRIX2			= 183
_VRUT_SETMATRIX3			= 184
_VRUT_SETMATRIX4			= 185
_VRUT_ADDOSGCHILD			= 186
_VRUT_ADDOSGCHILD			= 186
_VRUT_TRANSOSGCHILD			= 187
_VRUT_ROTOSGCHILD			= 188
_VRUT_SCALEOSGCHILD			= 189
_VRUT_CENTEROSGCHILD		= 190
_VRUT_CURTAINOSGCHILD		= 191
_VRUT_WIDTHTEXT				= 192
_VRUT_MESSAGETEXT			= 193
_VRUT_GETPICKED				= 194
_VRUT_NOPROJECTION			= 195


class OSGChild:
	
	def __init__(self):
		self.id = -1
		
	def translate(self,x,y,z):
		_ipcSend(_VRUT_TRANSOSGCHILD, self.id, 0, 'nil', x, y, z, 0)
		
	def rotate(self,x,y,z,deg=''):
		if deg == '':
			_ipcSend(_VRUT_ROTOSGCHILD, self.id, 0, 'nil', x, y, z, 0)
		else:
			_ipcSend(_VRUT_ROTOSGCHILD, self.id, 0, 'nil', x, y, z, deg)
			
	def scale(self,x,y,z):
		_ipcSend(_VRUT_SCALEOSGCHILD, self.id, 0, 'nil', x, y, z, 0)
		
	def curtain(self,state):
		_ipcSend(_VRUT_CURTAINOSGCHILD, self.id, state, 'nil', 0, 0, 0, 0)
		
	def center(self,x,y,z):
		_ipcSend(_VRUT_CENTEROSGCHILD, self.id, 0, 'nil', x, y, z, 0)
		
		
		
#************************************************************
# Callback events
KEYBOARD_EVENT		= 1
TIMER_EVENT			= 2
PLAYBACK_EVENT		= 3
HOTSPOT_EVENT		= 4
BUTTON_EVENT		= 5
MENU_EVENT			= 6
MOUSEBUTTON_EVENT	= 7
MOUSEMOVE_EVENT		= 8
MAIL_EVENT			= 9
COLLISION_EVENT		= 10
NETWORK_EVENT		= 11
KEYBOARDX_EVENT		= 12
DUMMY				= -1	# Terrible kluge for fact that list starts at 1 instead of 0



_VRUT_EVENT_LIST = KEYBOARD_EVENT, \
                  TIMER_EVENT, \
                  PLAYBACK_EVENT,\
                  HOTSPOT_EVENT, \
                  BUTTON_EVENT, \
                  MENU_EVENT, \
                  MOUSEBUTTON_EVENT, \
                  MOUSEMOVE_EVENT, \
                  MAIL_EVENT, \
                  COLLISION_EVENT, \
                  NETWORK_EVENT, \
                  KEYBOARDX_EVENT, \
                  DUMMY

_EventFunctionPointers = ['']*len(_VRUT_EVENT_LIST)
# This is used by _HandleGenericEvents

LI_NONE			   = 0
LI_LIGHTING_REAL   = 1
LI_LIGHTING_PSEUDO = 2
LI_NOMIPMAPS	   = 4
LI_RESET		   = 8
LI_TESSELATE	   = 16
LI_SPRITE		   = 32
LI_WIREFRAME	   = 64

KEY_F1							= '1001'
KEY_F2							= '1002'
KEY_F3							= '1003'
KEY_F4							= '1004'
KEY_F5							= '1005'
KEY_F6							= '1006'
KEY_F7							= '1007'
KEY_F8							= '1008'
KEY_F9							= '1009'
KEY_F10							= '1010'
KEY_F11							= '1011'
KEY_F12							= '1012'

KEY_UP							= '1101'
KEY_DOWN						= '1103'
KEY_LEFT						= '1100'
KEY_RIGHT						= '1102'

KEY_ESC							= str(27)

_MATRIX_CHILD					= 1
_MATRIX_VIEWPOINT				= 2



#************************************************************
# GL ENABLE and DISABLE COMMANDS (taken from gl.h)
# These are the #defs with the #define  GL_ stripped off

ALPHA_TEST                       = 0x0BC0
BLEND                            = 0x0BE2
CW                               = 0x0900
CCW                              = 0x0901
COLOR_MATERIAL                   = 0x0B57
CULL_FACE                        = 0x0B44
DEPTH_TEST                       = 0x0B71
FLAT                          	 = 0x1D00
FOG                              = 0x0B60
LIGHT0                           = 0x4000
LIGHT1                           = 0x4001
LIGHT2                           = 0x4002
LIGHT3                           = 0x4003
LIGHT4                           = 0x4004
LIGHT5                           = 0x4005
LIGHT6                           = 0x4006
LIGHT7                           = 0x4007
LIGHTING                         = 0x0B50
LINES                         	 = 0x0001
LINE_LOOP                     	 = 0x0002
LINE_STRIP                    	 = 0x0003
NORMALIZE                        = 0x0BA1
POINTS                        	 = 0x0000
POLYGON                       	 = 0x0009
POLYGON_SMOOTH                   = 0x0B41
POLYGON_STIPPLE                  = 0x0B42
QUADS                         	 = 0x0007
QUAD_STRIP                    	 = 0x0008
SCISSOR_TEST                     = 0x0C11
SMOOTH                        	 = 0x1D01
STENCIL_TEST                     = 0x0B90
TEXTURE_2D                       = 0x0DE1
TRIANGLES                     	 = 0x0004
TRIANGLE_FAN                  	 = 0x0006
TRIANGLE_STRIP                	 = 0x0005
POINT_SMOOTH			         = 0x0B10
LINE_SMOOTH			             = 0x0B20
EXP                              = 0x0800
EXP2                             = 0x0801
LINEAR                           = 0x2601


#************************************************************
# Initialize display modes

MONO			= 0
ORTHOGRAPHIC	= 1
CONSOLE			= 2
FULLSCREEN		= 256				# was 4
PERSPECTIVE		= 8
NICE			= 16
DOUBLEWIDTH		= 32
JERRY			= 64
DEBUG			= 128
HMD				= 256				
BETA			= 512
STEREO			= 1024
TRACKER			= 2048
PROMPT			= 4096
OPTION1			= 8192
OPTION2			= 16384
LCD_PROJECTOR	= 32768
STACK			= 32768
CRYSTAL         = 65536



# Get options

MAIN_WINDOWN	= '_MainWindow'
MAIN_VIEWPOINT	= '_MainViewpoint'





#************************************************************
# Render flags attached to layers

ENVIRONMENT_MAP		= 1
MODULATE			= 2
TEXGEN				= 4
TEXMIPMAP			= 8
TRANSPARENCY		= 16
LOCKLIGHTS			= 32
USEALPHA			= 64   # not used by Python, place holder
MIRROR				= 128
BLEND				= 256


#************************************************************
# Sensor types

MOUSE				= 1

#************************************************************
# Mouse position retrieval formats
ABSOLUTE             = 1
RELATIVE             = 2


#************************************************************
# These should be identical to those in vrut.h
HEAD				= 2
BODY				= 1
WORLD				= 0

HEAD_POS			= 4     # These values can 
HEAD_ORI			= 8	# be or'd together.  Must be
BODY_ORI			= 16	# different than anchor IDs
PERIPHERALS			= 32
USER_KNOB			= 64
REPLACE				= 128
ENDHOLD				= 256
INITMESG			= 512

XAXIS				= 0
YAXIS				= 1
ZAXIS				= 2

LEFT_WINDOW			= 1
RIGHT_WINDOW		= 2
LEFT_EYE			= 1
RIGHT_EYE			= 2

OPEN				= 0	# Curtain params
CLOSE				= 1

ON                  = 1 
OFF                 = 0
TOGGLE				= 2

LOOP				= 1
PAUSE				= 2


STOP				= 0	# Tracker params
START				= 1
RESET				= 2

SCROLL_PUSH			= 1
SCROLL_ITEM			= 2

RECTANGLE_HOTSPOT_IN		= 1
RECTANGLE_HOTSPOT_OUT		= 2	
CIRCLE_HOTSPOT_IN		= 3
CIRCLE_HOTSPOT_OUT		= 4
BOX_HOTSPOT_IN			= 5
BOX_HOTSPOT_OUT			= 6

# For compatibility with old scripts:
RECTANGLE_HOTSPOT		= 1
CIRCLE_HOTSPOT			= 3

HOTSPOT_LIST	= RECTANGLE_HOTSPOT_IN, \
				  RECTANGLE_HOTSPOT_OUT, \
				  CIRCLE_HOTSPOT_IN, \
				  CIRCLE_HOTSPOT_OUT, \
				  RECTANGLE_HOTSPOT, \
				  CIRCLE_HOTSPOT, \
				  BOX_HOTSPOT_IN,\
				  BOX_HOTSPOT_OUT

TRACK				= 1
NOTRACK				= 0

FASTEST_EXPIRATION	= 0.00001


#STANDALONE			0		# !! these need to be sync'd in vrut.h !!
#NETHOST				1
#NETCLIENT			2

# Add objects
VIEWPOINT			= 101
WINDOW				= 102
TEXT3D				= 103
MAILBOX				= 104


#************************************************************
# mouse buttons
# make sure these correspond to the values 
# used in frameLeft() in opt_context.cxx

MOUSEBUTTON_LEFT	= 1
MOUSEBUTTON_RIGHT	= 2

#************************************************************
#************************************************************


#************************************************************
# Keep track of the number of layers in each clump
# These will keep track of ATTEMPTS not SUCCESSFUL
# push or pop.  Right now I don't have a way for vrut.py
# to get feedback from VRE :(
#************************************************************
_numWorld = 0
_numBody = 0
_numHead = 0
_numTexture = 0
_numWindow = 1
_numViewpoint = 1
_alreadyGoing = 0

_numChild = 0
_numSensor = 0
_numRenderer = 0
_numTexture = 0
_numMail = 0
_numLights = 0


# Create a dictionary to keep track of all objects and textures
nameDict = {}


################################################################################################
################################################################################################
#									    Event Class											   #
################################################################################################
################################################################################################


#************************************************************
class EventClass:

	count = [0]
	timerEventFunction = ''
	_EventClassList = []

	def __init__(self):
		self.count[0] = self.count[0] + 1
		self.id = self.count[0]
		self._EventClassList.append(self)

	def callback(self, eventType, functionName):
		if type(functionName) == types.StringType:
			# Put a placeholder in the list of functionPointers so that
			# it'll be OK if the user mixes calling conventions (e.g., by
			# string name or by function pointer.
			_ipcSend(_VRUT_CLASSCALLBACK, self.id-1, eventType, functionName, 0, 0, 0, 0)

		elif type(functionName) == types.MethodType:
			self.timerEventFunction = functionName
			_ipcSend(_VRUT_CLASSCALLBACK, self.id-1, eventType, '', 0, 0, 0, 0)

		else:
			print '**ERROR: Invalid function', functionName

	def starttimer(self, timerNum = 1, timeSec = FASTEST_EXPIRATION, repeat = 0):
		# This self.id isn't supposed to have a -1!!!!
		_ipcSend(_VRUT_CLASSSTARTTIMER, self.id, timerNum, 'nil', timeSec, repeat, 0, 0);

	def handleEvent(self, eventType, data1):
		if eventType == TIMER_EVENT:
			self.timerEventFunction(data1)


#************************************************************
def _HandleGenericClassEvent(eventType, childID, data1):
	EventClass._EventClassList[childID].handleEvent(eventType, data1)



################################################################################################
################################################################################################
#											OSC												   #
################################################################################################
################################################################################################

# Callback event added for VRUT event ability
# Creates a new thread that monitors the specific port
# for incomming messages                     --Ryan Jaeger
InSocket = 0
OSC_PORT = 4950

OutSocket = socket.socket(socket.AF_INET, socket.SOCK_DGRAM)

class Mailbox:
	
	def __init__(self,address):
		self.address = address
		
	def send(self,*args):
		HOST = self.address
		message = OSCMessage()
		for x in args:
			message.append(x)
		try:
			OutSocket.sendto(message.getBinary(), (HOST, OSC_PORT))
		except:
			print 'Could not send message to \'',self.address,'\''

		
def addOSC(address):
	return Mailbox(address)
    
def OSCcallback(eventType, functionName):
	global callbackFunction, InSocket
	try: 
		if InSocket != 0:
			InSocket.close()
		InSocket = socket.socket(socket.AF_INET, socket.SOCK_DGRAM)
		InSocket.setsockopt(socket.SOL_SOCKET, socket.SO_REUSEADDR, 1)
		InSocket.bind(('', OSC_PORT))

		callbackFunction = functionName
		OSCThread = MyThread()
		OSCThread.start()
	except:
		print '** Error: Failed to establish network connection'
	
	
# Creates an instance of a thread that only monitors a given port for messages
class MyThread(threading.Thread,EventClass):

	def networkTimer(self,num):
		pass
		
	def __init__(self):
		EventClass.__init__(self)
		self._stopevent = threading.Event()
		threading.Thread.__init__(self, name="MyThread")
		self.callback(TIMER_EVENT,self.networkTimer)
		self.starttimer(0,0.01,-1)
        
	def run(self): #main thread
		while 1:
			#hangs out here until a message is recieved
			data = InSocket.recv(1024)
			if data:
				message = decodeOSC(data)
				#print 'recieved a message', message
				if message:
					callbackFunction(message)
				
				
def send(host, message):
	HOST = host
	# sends a message in OSC format to by typecasted apon recieving it
	OutSocket.sendto(message, (HOST, PORT))

# Modified original OSC messaging code
def hexDump(bytes):
#	Useful utility; prints the string in hexadecimal
	for i in range(len(bytes)):
		sys.stdout.write("%2x " % (ord(bytes[i])))
		if (i+1) % 8 == 0:
			print repr(bytes[i-7:i+1])

	if(len(bytes) % 8 != 0):
		print string.rjust("", 11), repr(bytes[i-7:i+1])


class OSCMessage:
#	Builds typetagged OSC messages.
	def __init__(self):
		self.address  = ""
		self.typetags = ","
		self.message  = ""

	def setAddress(self, address):
		self.address = address

	def append(self, argument, typehint = None):
#		Appends data to the message,
#		updating the typetags based on
#		the argument's type.
#		If the argument is a blob (counted string)
#		pass in 'b' as typehint.

		if typehint == 'b':
			binary = OSCBlob(argument)
		else:
			binary = OSCArgument(argument)

		self.typetags = self.typetags + binary[0]
		self.rawAppend(binary[1])

	def rawAppend(self, data):
#		Appends raw data to the message.  Use append().
		self.message = self.message + data

	def getBinary(self):
#		Returns the binary message (so far) with typetags.
		address  = OSCArgument(self.address)[1]
		typetags = OSCArgument(self.typetags)[1]
		#return self.message
		return address + typetags + self.message

	def __repr__(self):
		return self.getBinary()


def readString(data):
	length   = string.find(data,"\0")
	nextData = int(math.ceil((length+1) / 4.0) * 4)
	return (data[0:length], data[nextData:])


def readBlob(data):
	length   = struct.unpack(">i", data[0:4])[0]    
	nextData = int(math.ceil((length) / 4.0) * 4) + 4   
	return (data[4:length+4], data[nextData:])


def readInt(data):
	if(len(data)<4):
		print "Error: too few bytes for int", data, len(data)
		rest = data
		integer = 0
	else:
		integer = struct.unpack(">i", data[0:4])[0]
		rest    = data[4:]
        
	return (integer, rest)



def readLong(data):
#	Tries to interpret the next 8 bytes of the data
#	as a 64-bit signed integer.
	high, low = struct.unpack(">ll", data[0:8])
	big = (long(high) << 32) + low
	rest = data[8:]
	return (big, rest)



def readFloat(data):
	if(len(data)<4):
		print "Error: too few bytes for float", data, len(data)
		rest = data
		float = 0
	else:
		float = struct.unpack(">f", data[0:4])[0]
		rest  = data[4:]

	return (float, rest)


def OSCBlob(next):
#	Convert a string into an OSC Blob,
#	returning a (typetag, data) tuple.

	if type(next) == type(""):
		length = len(next)
		padded = math.ceil((len(next)) / 4.0) * 4
		binary = struct.pack(">i%ds" % (padded), length, next)
		tag    = 'b'
	else:
		tag    = ''
		binary = ''
    
	return (tag, binary)


def OSCArgument(next):
#	Convert some Python types to their
#	OSC binary representations, returning a
#	(typetag, data) tuple.
    
	if type(next) == type(""):        
		OSCstringLength = math.ceil((len(next)+1) / 4.0) * 4
		binary  = struct.pack(">%ds" % (OSCstringLength), next)
		tag = "s"
	elif type(next) == type(42.5):
		binary  = struct.pack(">f", next)
		tag = "f"
	elif type(next) == type(13):
		binary  = struct.pack(">i", next)
		tag = "i"
	else:
		binary  = ""
		tag = ""

	return (tag, binary)


def parseArgs(args):
#	Given a list of strings, produces a list
#	where those strings have been parsed (where
#	possible) as floats or integers.
	parsed = []
	for arg in args:
		print arg
		arg = arg.strip()
		interpretation = None
		try:
			interpretation = float(arg)
			if string.find(arg, ".") == -1:
				interpretation = int(interpretation)
		except:
			# Oh - it was a string.
			interpretation = arg
			pass
		parsed.append(interpretation)
	return parsed

def decodeOSC(data):
#	Converts a typetagged OSC message to a Python list.
	table = {"i":readInt, "f":readFloat, "s":readString, "b":readBlob}
	decoded = []
	address,  rest = readString(data)
	typetags = ""

	if address == "#bundle":
		time, rest = readLong(rest)
		decoded.append(address)
		decoded.append(time)
		while len(rest)>0:
			length, rest = readInt(rest)
			decoded.append(decodeOSC(rest[:length]))
			rest = rest[length:]

	elif len(rest)>0:
		typetags, rest = readString(rest)
		decoded.append(address)
		decoded.append(typetags)
		if(typetags[0] == ","):
			for tag in typetags[1:]:
				value, rest = table[tag](rest)                
				decoded.append(value)
		else:
			print "Oops, typetag lacks the magic ,"

	return decoded


class CallbackManager:
#	This utility class maps OSC addresses to callables.
#
#	The CallbackManager calls its callbacks with a list
#	of decoded OSC arguments, including the address and
#	the typetags as the first two arguments.

	def __init__(self):
		self.callbacks = {}
		self.add(self.unbundler, "#bundle")

	def handle(self, data, source = None):
#		Given OSC data, tries to call the callback with the
#		right address.
		decoded = decodeOSC(data)
		self.dispatch(decoded)

	def dispatch(self, message):
#		Sends decoded OSC data to an appropriate calback
		try:
			address = message[0]
			self.callbacks[address](message)
		except KeyError, e:
			# address not found
			print 'foo'
			pass
		except None, e:
			print "Exception in", address, "callback :", e

		return

	def add(self, callback, name):
#		Adds a callback to our set of callbacks,
#		or removes the callback with name if callback
#		is None.
		if callback == None:
			del self.callbacks[name]
		else:
			self.callbacks[name] = callback

	def unbundler(self, messages):
#		Dispatch the messages in a decoded bundle.
		# first two elements are #bundle and the time tag, rest are messages.
		for message in messages[2:]:
			self.dispatch(message)


################################################################################################
################################################################################################
#										Vizard Functions									   #
################################################################################################
################################################################################################


def input(prompt, value=''):

	if _where == 'INT':
		return 0

	retval = vizinput.input(prompt,value)
	try:
		num = string.atoi(retval)
		return num
	except:
		try:
			num = string.atof(retval)
			return num
		except:
			return retval

def choose(prompt,choices, selection=0):

	if _where == 'INT':
		return 0
	
	return vizinput.choose(prompt,choices,selection)


#************************************************************
def fog(start, end = -1):

	if end != -1:
		mode = LINEAR
		density = 1
	else:
		mode = EXP
		density = start

	_ipcSend(_VRUT_FOG, mode, 0, 'nil', start, end, density, 0)


#************************************************************
def clearcolor(red, green, blue):

	_ipcSend(_VRUT_CLEARCOLOR, 0, 0, 'nil', red, green, blue, 0)


#************************************************************
def pushlayer(layerType, fileName, option1 = -99, option2 = -99):
   
	global _numWorld, _numBody, _numHead

	if layerType == HEAD:
		#print 'Head type'
		_numHead = _numHead +1
		nameDict[fileName] = _numHead
		#execute('vrut._numHead = vrut._numHead +1')
		#execute("vrut.nameDict['"+fileName+"'] = vrut._numHead")
	elif layerType == BODY:
		#print 'Body type'
		_numBody = _numBody +1
		nameDict[fileName] = _numBody
		#execute('vrut._numBody = vrut._numBody +1')
		#execute("vrut.nameDict['"+fileName+"'] = vrut._numBody")
	elif layerType == WORLD:
		#print 'World type'
		_numWorld = _numWorld +1
		nameDict[fileName] = _numWorld
		#execute('vrut._numWorld = vrut._numWorld +1')
		#execute("vrut.nameDict['"+fileName+"'] = vrut._numWorld")
	else:
		print 'Unknown type'
		return


	# For now, alway set pseudolighting flag
	if option1 == -99:
		option1 = LI_LIGHTING_PSEUDO
	else:
		option1 = option1 | LI_LIGHTING_PSEUDO

	if option1 & LI_TESSELATE:
		if option2 == -99:
			print '** Tesselation level must be specified'
			option1 = option1 ^ LI_TESSELATE
		elif option2 > 10:
			print '** Tesselation level too high'
		option1 = option1 ^ LI_TESSELATE

	if option1 & LI_LIGHTING_PSEUDO:
		if option2 == -99:
			option2 = 1.0

	if option1 & LI_LIGHTING_REAL:
		if option2 == -99:
			option2 = 1.0

	_ipcSend(_VRUT_PUSHLAYER, layerType, option1, fileName, option2, 0, 0, 0)


#************************************************************
def poplayer(layerType):

	global _numWorld, _numBody, _numHead

	if layerType == HEAD:
		print 'Head type'
		_numHead = _numHead -1
	elif layerType == BODY:
		print 'Body type'
		_numBody = _numBody -1
	elif layerType == WORLD:
		print 'World type'
		_numHead = _numHead -1
	else:
		print 'Unknown type'
		return

	_ipcSend(_VRUT_POPLAYER, layerType, 0, 'nil', 0,0,0,0)




#************************************************************
def tracker(command = START, device = 0):

	if command == START:
		_ipcSend(_VRUT_STARTTRACKING, device, 0, 'nil', 0, 0, 0, 0)

	elif command == STOP:
		_ipcSend(_VRUT_CLOSETRACKER, 0, 0, 'nil', 0, 0, 0, 0);

	else:
		print '** Unknown tracker option!'



#************************************************************
def go(mode = CONSOLE, moduleName = '', texturesPredominate = 0):
	
	global _numWorld, _numBody, _numHead, _numTexture, nameDict, _numWindow, _numViewpoint
	global _numChild, _numSensor, _numTexture, _numMail, _numRenderer, _numLights

	global _ipcSend, _ipcSendHold, _where

	#print 'go issued with:', _where, ipc.getalive()

	if ipc.getalive() == 0:
		_where = 'INT'
		
	else:
		# Check that winviz really seems to be running...
		if _where == 'INT' and time.time() - ipc.getecho() > 2:
			# it's crashed
			#print 'Crash detected--Resetting'
			ipc.setalive(0)	
			_where = 'INT'	
		
	
	# This is a fix so that when VRUT was not properly shutdown, it 
	# will recognize the fact and reset itself.
	#if ipc.getalive() == 2:
	#	print '*** RESETING Vizard ***'
	#	ipc.setalive(0)

	if ipc.getalive() == 3:
		# I think this means we're QuickFreshing
		# print 'got 3, returning from top'
		return


	ipc.setalive(ipc.getalive() + 1)

	if _where == 'X':
		#print 'already running..., returning'					
		return


	# If winvrut is already running, skip the go command.
	#if ipc.getalive() == 1:
	#	   return
	#else:
	#   ipc.setalive(1)


	_numWorld = 0
	_numBody = 0
	_numHead = 0
	_numTexture = 0
	_numWindow = 1
	_numViewpoint = 1
	nameDict = {}

	# These are used to maintain a local list of object IDs which
	# allows the users to access these objects from the interactive
	# window.  
	_numChild = 0
	_numSensor = 0
	_numRenderer = 0
	_numTexture = 0
	_numWindow = 1
	_numViewpoint = 1
	_numMail = 0
	_numLights = 0

	import win32api

	if ipc.getalive() < 3:
		ipc.sendnow(_VRUT_INITDISPLAY, mode, 0, os.getcwd(), 0, 0, 0, 0);

	# the command gets put in a queue, and is accessed
	# by vrutIPCInit() [in vrut_ipc.c()] on startup

	# If the user passed a moduleName, this the user wishes that this
	# module be imported into the internal python interpreter.  Do this
	# by issuing an execute command.
	# This new method effectively replaces having to split VRUT programs
	# into a main and a callback program and will also largely eliminate 
	# the use of vrut.execute by the user.

	# The moduleName method is now obsolete and it not used.  It has been
	# replaced by the auto find module name code below that someone on the
	# python newsgroup described to me.  The trick uses exeption handling, 
	# which can then peek on the traceback stack to find out who's been
	# calling who.


	# Obsolete except for maintain backward compatibility
	if moduleName != '':
		ipc.setalive(3)
		ipc.sendnow(_VRUT_EXECPYTHON, 0, 0, 'import ' +moduleName, 0, 0, 0, 0);
		#execute('import ' + moduleName)

	else:
		ok = 0
		double = 0
		#################################################
		# AUTO FIND MODULE NAME
		#################################################

		try:
			raise None
		except:
			c = sys.exc_traceback.tb_frame.f_back.f_code
			#c2 = sys.exc_traceback.tb_frame.f_back.f_back.f_code
			#print '----', c.co_filename, '-----', c.co_name

			try:
				c2 = sys.exc_traceback.tb_frame.f_back.f_back.f_code
			except:
				pass
				#print '-- Launched outside Pythonwin --'
				#ipc.setalive(4)
				#double = 1


		#print '***** ', c.co_filename, c.co_name


		if c.co_filename != '<interactive input>':

			if os.path.isfile(c.co_filename):
				runstring = 'execfile("' + c.co_filename +'")'
				runstring = string.replace(runstring, '\\', '\\\\')
				#print 'runstring: ', runstring
				#execute(runstring)
				#print 'DOUBLE: ', double, ipc.getalive()
				if double == 0:
					ipc.sendnow(_VRUT_EXECPYTHON, 0, 0, runstring, 0, 0, 0, 0);

			else:
				#print 'trying 2...'
				for path in sys.path:
					tryPath = path + '\\' + c.co_filename
					#tryPath = string.replace(tryPath, '\\', '\\\\')
					#print tryPath
					if os.path.isfile(tryPath):
						runstring = 'execfile("' + tryPath +'")'
						runstring = string.replace(runstring, '\\', '\\\\')
						#execute(runstring)
						ipc.sendnow(_VRUT_EXECPYTHON, 0, 0, runstring, 0, 0, 0, 0);
						#print "*** ran: ", runstring
						ok = 1
						break   

				if ok != 1:
					pass
					#print "** Couldn't find a module by that name!"

		else: # This is an interactive session
			ipc.setalive(2)	

		#################################################

	if ipc.getalive() == 3:
		# 12/16/02 - I'm moving these up here because QuickFresh wasn't working
		# if any interactive mode had been used, inluding stayontop button, but the
		# user.
		_ipcSend = _dummySend
		_ipcSendHold = _dummySendHold
		#print 'got 3, returning from bottom'
		return



	elif mode & BETA:
		if texturesPredominate:
			win32api.WinExec(_BIN_BETA+' 1') 
		else: 
			win32api.WinExec(_BIN_BETA) 

		win32api.Sleep(500)

	elif not mode & DEBUG:
		if texturesPredominate:
			win32api.WinExec(_BIN_PRIMARY+' 1') 
		else: 
			win32api.WinExec(_BIN_PRIMARY) 

		win32api.Sleep(500)              # seems necessary for textures to get loaded when running a script

	if mode & NICE:
		_repositionwindow(1000,30)

	if mode & LCD_PROJECTOR:
		_repositionwindow(1,436)


	_ipcSend = _dummySend
	_ipcSendHold = _dummySendHold

	#os.chdir(curdir)
	#sys.tracebacklimit = 0
	#sys.exit()
	#ipc.send = _dummySend
	#ipc.sendHold = _dummySendHold


#************************************************************
def enable(mode):

	_ipcSend(_VRUT_GL_ENABLE, mode, 0, 'nil', 0, 0, 0, 0);
	

#************************************************************
def disable(mode):

	_ipcSend(_VRUT_GL_DISABLE, mode, 0, 'nil', 0, 0, 0, 0);
	

#************************************************************
def execute(command):
	
	_ipcSend(_VRUT_EXECPYTHON, 0, 0, command, 0, 0, 0, 0);





_DIRECTOR_SCRIPT = 0
#************************************************************
def _director():

	print 'Initiating director script'
	_DIRECTOR_SCRIPT()
	print 'Completed director script'


#************************************************************
def director(command):
	global _DIRECTOR_SCRIPT

	#_ipcSend(_VRUT_EXECPYTHON2, 0, 0, command, 0, 0, 0, 0);

	_DIRECTOR_SCRIPT = command
	_ipcSend(_VRUT_EXECPYTHON2, 0, 0, 'viz._director()', 0, 0, 0, 0)



#************************************************************
def callback(eventType, functionName):
	
	if eventType in _VRUT_EVENT_LIST:
		if ipc.getalive() >= 2:  # if not old module name convention
			# For backward compatibility with __name__ usage, need to 
			# strip off the "__name__." portion, unless it belongs to a 
			# child script.

			# The following ended up breaking the old style __name__ conventions
			# If this is a child script, then the parent module name will start
			# with a period, as in './modulename'.
			##s = string.split(functionName, '.')
			##if len(s) == 2:
			##	functionName = s[1]   # take only portion after the period

			if eventType == NETWORK_EVENT:
				OSCcallback(eventType,functionName)
			elif type(functionName) == types.StringType:
				# VRUT 2.x calling convention
				_ipcSend(_VRUT_REGISTERCALLBACK, eventType, 0, functionName, 0, 0, 0, 0);
				if ((eventType == MOUSEBUTTON_EVENT) or (eventType == MOUSEMOVE_EVENT)):
					_ipcSend(_VRUT_MOUSECONSTRAIN, 1, 0, 'nil', 0, 0, 0, 0)

			elif type(functionName) == types.FunctionType:
				# Viz 1.x calling convention
				_ipcSend(_VRUT_REGISTERCALLBACK, eventType, 0, '', 0, 0, 0, 0);
				_EventFunctionPointers[eventType] = functionName

			else:
				print '**ERROR: Invalid function name or pointer'
			


	else:
		print 'INVALID EVENT TYPE'


#************************************************************
def starttimer(timerNum=1, timeSec = FASTEST_EXPIRATION, repeat = 0):
	
	_ipcSend(_VRUT_STARTTIMER, timerNum, 0, 'nil', timeSec, repeat, 0, 0);

#************************************************************
def killtimer(timerNum=1):
	_ipcSend(_VRUT_KILLTIMER, timerNum, 0, 'nil', 0, 0, 0, 0)

	

#************************************************************
def starthotspot(hotspotNum, kind, xloc, zloc,  p1, p2 = 0, p3 = 0, yloc=0):

	# Negate Z center value so postitive Z is North bound
	list = xloc, zloc, p1, p2, p3, yloc
	mesg = repr(list)

	if kind not in HOTSPOT_LIST:
	  print '** Invalid hotspot kind.'
	  return

	# -1 means that this hotspot's source is the eye position
	_ipcSend(_VRUT_STARTHOTSPOT, hotspotNum, kind, mesg, -1, 0, 0, 0)



#************************************************************
_currentType = -1
def startlayer(kind):
	
	global _currentType

	# Right now is always opens this layer on the 
	# world clump.	
	if _currentType == -1:   # ho
		_ipcSend(_VRUT_OPENLAYER, WORLD, 0, 'nil', 0, 0, 0, 0)
		_ipcSend(_VRUT_BEGINLIST, kind, 0, 'nil', 0, 0, 0, 0)
		_currentType = kind

	elif _currentType != kind:
		_ipcSend(_VRUT_ENDLIST, 0, 0, 'nil', 0, 0, 0, 0)
		_ipcSend(_VRUT_BEGINLIST, kind, 0, 'nil', 0, 0, 0, 0)
		_currentType = kind

	_currentType = kind


#************************************************************
def endlayer():
	
	global _currentType

	# Only end if a list has already been started
	if _currentType != -1:
		_currentType = -1	
		_ipcSend(_VRUT_ENDLIST, 0, 0, 'nil', 0, 0, 0, 0)
		_ipcSend(_VRUT_CLOSELAYER, WORLD, 0, 'nil', 0, 0, 0, 0)



#************************************************************
def vertex(x, y, z):

	if _currentType != -1:
		_ipcSend(_VRUT_VERTEX3D, 0, 0, 'nil', x, y, -z, 0)
	else:
		print 'USE VIZ.STARTLAYER BEFORE VIZ.VERTEX'



#************************************************************
def rectangle(top, left, bottom, right):

	startlayer(QUADS)	
	vertex(left, top, 0)
	vertex(left, bottom, 0)
	vertex(right, bottom, 0)
	vertex(right, top, 0)


#************************************************************
def vertexcolor(red, green, blue):
	
	_ipcSend(_VRUT_VERTEXCOLOR, 0, 0, 'nil', red, green, blue, 0)

#************************************************************
def pointsize(numPixels):
	
	_ipcSend(_VRUT_POINTSIZE, 0, 0, 'nil', numPixels, 0, 0, 0)


#************************************************************
def linewidth(numPixels):
	_ipcSend(_VRUT_LINEWIDTH, 0, 0, 'nil', numPixels, 0, 0, 0)


#************************************************************
#************************************************************
def reset(what, layerNum = -99):

	if what == HEAD_POS:
		_ipcSend(_VRUT_RESETVIEWT, 0, 0, 'nil', 0, 0, 0, 0)

	elif what == BODY_ORI:
		_ipcSend(_VRUT_RESETVIEWR, 0, 0, 'nil', 0, 0, 0, 0)

	else:
		if layerNum == -99:
			print 'Bad reset target!'

		else:
			_ipcSend(_VRUT_RESETLAYER, what, layerNum, 'nil', 0, 0, 0, 0)



#************************************************************
def curtain(mode, what=999, layer=999, option = 999):
 
	if what == 999 and layer == 999:
		_ipcSend(_VRUT_SCREEN, mode, 0, 'nil', 0, 0, 0, 0)
	elif what == LEFT_EYE and layer == 999:
		_ipcSend(_VRUT_CURTAINEYE, LEFT_EYE, mode^1, 'nil', 0, 0, 0, 0)
	elif what == RIGHT_EYE and layer == 999:
		_ipcSend(_VRUT_CURTAINEYE, RIGHT_EYE, mode^1, 'nil', 0, 0, 0, 0)
	elif option != 999:
		_ipcSend(_VRUT_TOGGLE_LAYEREYE, mode^1, what, 'nil', layer-1, option, 0, 0)
	else:
		_ipcSend(_VRUT_TOGGLE_LAYER, mode^1, layer-1, 'nil', what, 0, 0, 0)


#************************************************************
# With four parameters, translate offsets the viewing position.
# With five parameters, it offsets a model layer.  The variable
# names didn't work out well for the 4 param version since this
# 'overloading' was an afterthought.
#************************************************************
def translate(kind, layerNum, x, y, z=0):

	if type(kind) == types.InstanceType:
		# We have a child
		# x <- layerNum     y <- x     z <- y
		_ipcSend(_VRUT_TRANSLATECHILDR, kind.id-1, 0, 'nil', layerNum, x, y, 0)

	elif kind == HEAD_POS:
		_ipcSend(_VRUT_OFFSETVIEWT, 0, 0, 'nil', layerNum, x, -y, 0)

	else:
		_ipcSend(_VRUT_TRANSLATE, kind, layerNum, 'nil', x, y, -z, 0)

#************************************************************
# This is another 'overloaded' function that was originally 
# written with the goal of rotating a model layer.  If instead
# of a clump anchor HEAD_ORI is passed, the remaining three
# parameters become: yaw, pitch, roll.
#************************************************************
# Axis: XAXIS, YAXIS, ZAXIS; Angle in degrees
def rotate(kind, layerNum, axis, angle=0):

	if type(kind) == types.InstanceType:
		# We have a child
		# axis <- layerNum     deg <- axis
		_ipcSend(_VRUT_ROTATECHILDR, kind.id-1, layerNum, 'nil', axis, 0, 0, 0)

	elif kind == BODY_ORI:
		yaw = layerNum
		pitch = axis
		roll = angle
		_ipcSend(_VRUT_OFFSETVIEWR, 0, 0, 'nil', yaw, pitch, roll, 0)
	else:
		if axis in range(0,3):
			_ipcSend(_VRUT_ROTATE, kind, layerNum, 'nil', axis, -angle, 0, 0)
		else:
			print 'Invalid axis (use viz.XAXIS, YAXIS, ZAXIS)'


#************************************************************
def scale(kind, layerNum, x, y, z):

	_ipcSend(_VRUT_SCALE, kind, layerNum, 'nil', x, y, z, 0)


#************************************************************
def quit():

	_ipcSend(_VRUT_QUIT, 0, 0, 'nil', 0, 0, 0, 0)

	# Just in case winvrut shutdown somehow without signaling 
	# it's dead.
	ipc.setalive(0)


#************************************************************
def clearall():

	global _numWorld, _numBody, _numHead, _numTexture, nameDict, _numWindow, _numViewpoint

	_numWorld = 0
	_numBody = 0
	_numHead = 0
	_numTexture = 0
	_numWindow = 1
	_numViewpoint = 1
	nameDict = {}

	_ipcSend(_VRUT_CLEARALL, 0, 0, 'nil', 0, 0, 0, 0)


#************************************************************
def record(fileName, what, maxTime = 30):

	print '** This function not implemented yet'
	return

	_ipcSend(_VRUT_DATAFILTER, 0, 0, what, 0, 0, 0, 0)
	_ipcSend(_VRUT_RECORD, 0, 0, fileName, maxTime, 0, 0, 0)


#************************************************************
# callback is optional
def playback(fileName, what, callback = None):

	if callback != None:
		vrut.callback(_VRUT_PLAYBACK, callback)

	_ipcSend(_VRUT_DATAFILTER, what, 0, 'nil', 0, 0, 0, 0)
	_ipcSend(_VRUT_PLAYBACK, 0, 0, fileName, 0, 0, 0, 0)


#************************************************************
# hiding this command
def _select(target):
   
	print '** This function not implemented yet'
	return

	_ipcSend(_VRUT_SELECT, target, 0, 'nil', 0, 0, 0, 0)



#************************************************************
def pushscroll(layerType, list):

	global _numWorld, _numBody, _numHead

	if layerType == HEAD:
		print 'Head type'
		_numHead = _numHead +1
	elif layerType == BODY:
		print 'Body type'
		_numBody = _numBody +1
	elif layerType == WORLD:
		print 'World type'
		_numWorld = _numWorld +1
	else:
		print 'Unknown type'
		return

	if len(list) > 0:
		_ipcSend(_VRUT_SCROLLLIST, SCROLL_PUSH, layerType, 'nil', 0, 0, 0, 0)

		for mesg in list:
			_ipcSend(_VRUT_SCROLLLIST, SCROLL_ITEM, 0, mesg, 0, 0, 0, 0)



#************************************************************
def clamp(what, p1, p2, p3):

	_ipcSend(_VRUT_CLAMP, what, 0, 'nil', p1, p2, -p3, 0)


#************************************************************
# hiding this command
def _dataevent():

	_ipcSend(_VRUT_TRIGGERDATA, 0, 0, 'nil', 0, 0, 0, 0)


#************************************************************
def fov(vertFOV, h2vRatio):

	_ipcSend(_VRUT_SETFOV, 0, 0, 'nil', vertFOV, h2vRatio, 0, 0)

def setfov(vertFOV, h2vRatio):

	fov(vertFOV,h2vRatio)
	



#************************************************************
def pushtexture(fileName, options = 0):

	global _numTexture

	_ipcSend(_VRUT_PUSHTEXTURE, options, 0, fileName, 0, 0, 0, 0)

	_numTexture = _numTexture + 1
	nameDict[fileName] = _numTexture
	#execute('vrut._numTexture = vrut._numTexture +1')
	#execute("vrut.nameDict['"+fileName+"'] = vrut._numTexture")

#************************************************************
def bindtexture(layerType, layer, textureID, options = 0):

	global _numWorld, _numBody, _numHead, _numTexture

	if layerType == HEAD:
		if layer > 0 and layer <= _numHead and textureID <= _numTexture:
			okT = 1

	elif layerType == BODY:
		if layer > 0 and layer <= _numBody and textureID <= _numTexture:
			okT = 1

	elif layerType == WORLD:
		if layer > 0 and layer <= _numWorld and textureID <= _numTexture:
			okT = 1

	else:
		print 'Unknown type'
		return

	_ipcSend(_VRUT_BINDTEXTURE, layerType, layer, 'nil', textureID, options, 0, 0)

   
#************************************************************
# hiding this command
def _lighttest(on):

	_ipcSend(_VRUT_LIGHTTEST, on, 0, 'nil', 0, 0, 0, 0)


#************************************************************
def ipd(meters):

	_ipcSend(_VRUT_SETIPD, 0, 0, 'nil', meters, 0, 0, 0)
	
def setipd(meters):

	ipd(meters)



#************************************************************
def viewdist(meters):
	_ipcSend(_VRUT_SETVIEWDIST, 0, 0, 'nil', meters, 0, 0, 0)
	
def setviewdist(meters):
	viewdist(meters)



#************************************************************
def get(option):

	if option == MAIN_VIEWPOINT:
		view = VRUTViewpoint()
		view.id = 1
		return view

	else:
		return(ipc.receive(option))


#************************************************************
# This function is meant for internal use only and not to be
# part of the general VRUT API.
def _repositionwindow(xloc, yloc):

	_ipcSend(_VRUT_REPOSITION_WINDOW, xloc, yloc, 'nil', 0, 0, 0, 0)


#************************************************************
def lightenable(anchor, lightNum):

	_ipcSend(_VRUT_LIGHTENABLE, lightNum-1, anchor, 'nil', 0, 0, 0, 0)
 

#************************************************************
def lightdisable(lightNum):

	_ipcSend(_VRUT_LIGHTDISABLE, lightNum-1, 0, 'nil', 0, 0, 0, 0)
  

#************************************************************
def lightintensity(lightNum, brightness):

	_ipcSend(_VRUT_LIGHTINTENSITY, lightNum-1, 0, 'nil', brightness, 0, 0, 0)


#************************************************************
def lightambient(r, g, b):

	_ipcSend(_VRUT_LIGHTAMBIENT, 0, 0, 'nil', r, g, b, 0)

#************************************************************

def lightposition(lightNum, x, y, z):

	_ipcSend(_VRUT_LIGHTPOSITION, 0, lightNum-1, 'nil', x, y, z, 0)

#************************************************************
def lightcolor(lightNum, r, g, b):

	_ipcSend(_VRUT_LIGHTCOLOR, lightNum-1, 0, 'nil', r, g, b, 0)

#************************************************************
def lightspread(lightNum, angle):

	_ipcSend(_VRUT_LIGHTSPREAD, lightNum-1, 0, 'nil', angle, 0, 0, 0)

#************************************************************
def lightdirection(lightNum, x, y, z):

	_ipcSend(_VRUT_LIGHTDIRECTION, lightNum-1, 0, 'nil', x, y, z, 0)



#************************************************************
# hiding this command
def _pushground(width, length, cycles, div, tex):
	global _numWorld

	_numWorld = _numWorld +1
	_ipcSend(_VRUT_PUSHGROUND, div, cycles, 'nil', width, length, tex, 0)


#************************************************************
# hiding this command
def _groundbody(z):

	_ipcSend(_VRUT_GROUNDBODY, 0, 0, 'nil', z, 0, 0, 0)


#************************************************************
def framerate(status = 1):

	if status == NICE:
		status = status + 1

	_ipcSend(_VRUT_ECHOFRAMERATE, status, 0, 'nil', 0, 0, 0, 0)


#************************************************************
# Specify an anchor and layer, and then you'll have shadows.
# To turn off shadowing, pass optional vrut.FALSE as 3rd param.
#
# hiding this command
def _shadow(kind, layer, slant = .1, status = 1):

	if slant == 0:
		status = 0	 
	_ipcSend(_VRUT_SHADOW, kind, layer, 'nil', status, slant, 0, 0)


#************************************************************
def clip(near, far):

	_ipcSend(_VRUT_CLIP, 1, 0, 'nil', near, far, 0, 0)

def setclip(near, far):

	clip(near,far)


#************************************************************
def eyeheight(height, track = 0):

	_ipcSend(_VRUT_EYEHEIGHT, track, 0, 'nil',height, 0, 0, 0)


#************************************************************
def deletehotspot(id):

	_ipcSend(_VRUT_DELETEHOTSPOT, id, 0, 'nil', 0, 0, 0, 0)

#************************************************************
def pushattributes():

	_ipcSend(_VRUT_PUSHATTRIBUTES, 0, 0, 'nil', 0, 0, 0, 0)


#************************************************************
def popattributes():

	_ipcSend(_VRUT_POPATTRIBUTES, 0, 0, 'nil', 0, 0, 0, 0)


#************************************************************
def showmessage(mess,x=0,y=0,z=0,scale = 2, clump = 0):

	_ipcSend(_VRUT_SHOWMESSAGE, clump, scale, mess, x,y,z, 0)


#************************************************************
def messagecolor(messnum, r=1,g=1,b=1):

	_ipcSend(_VRUT_MESSAGECOLOR, messnum, 0, 'nil', r,g,b, 0)



#************************************************************
def material(clump, layer, x=0,y=0,z=0):

	_ipcSend(_VRUT_MATERIAL, clump, layer-1, 'nil', x,y,z, 0)

#************************************************************
def readpixel(x, y, filename = 'nil',  brt = -1):

	_ipcSend(_VRUT_READPIXEL, x, y, filename, brt,0,0, 0)


#############################################################
## ADDED at MIT  7/98



#************************************************************
#************************************************************
class PlayMediaObj:

	playmediaObj = 0
	video = 0
	filename = ''

	def play(self):
		self.playmediaObj.command(2)

	def pause(self):
		self.playmediaObj.command(3)

	def rewind(self):
		self.playmediaObj.command(4)

	def volume(self, volume):
		self.playmediaObj.command(6,'',volume)

	def jumpto(self, time):

		self.playmediaObj.command(7,'',time)
		
	def rate(self, rate):
		self.playmediaObj.command(8,'',rate)
		
	def loop(self, loop):
		self.playmediaObj.command(5,'',loop)
		
	def stoptime(self, time):
		self.playmediaObj.command(9,'',time)
		
	def getlength(self):
		self.playmediaObj.command(10)
		return self.playmediaObj.get()[1]

#************************************************************
class VRUTTexture:
	id = -1

	def texture(self, subObjectName):
		# pass -1 thru access2 so that the effect cover entire root
		_ipcSend(_VRUT_APPLY_TEXTURE, self.id-1, -1, subObjectName, 0, 0, 0, 0)


#************************************************************
def addtexture(fileName):

	global _numTexture
	
	if type(fileName) == types.InstanceType:
		# We have a texgen object (presumably)
		texture = VRUTTexture()
		texture.id = _ipcSendHold(_VRUT_ADD_TEXGEN, fileName.id, 0, 'nil', 0, 0, 0, 0)

	else:
		texture = VRUTTexture()
		texture.id = _ipcSendHold(_VRUT_ADD_TEXTURE, 0, 0, fileName, 0, 0, 0, 0)


	if texture.id >= 0:
		return texture
	elif texture.id == 0:
		print '** Texture not found!: ', fileName
	elif texture.id == -1:
		_numTexture = _numTexture + 1
		texture.id = _numTexture
		return texture

#************************************************************


#************************************************************
class VRUTChild:
	id = -1

	##	translation = [0.0, 0.0, 0.0]
	##	rotation = [0.0,1.0,0.0,0.0]
	
	def translate(self, x, y, z):
		_ipcSend(_VRUT_TRANSLATECHILD, self.id-1, 0, 'nil', x, y, z, 0)

	def rotate(self, x, y, z, deg=''):
		if deg == '':
			_ipcSend(_VRUT_ROTATECHILDEULER, self.id-1, 0, 'nil', x, y, z, 0)
		else:
			_ipcSend(_VRUT_ROTATECHILD, self.id-1, 0, 'nil', x, y, z, deg)

	def rotate2(self, x, y, z):
		_ipcSend(_VRUT_ROTATECHILDEULER2, self.id-1, 0, 'nil', x, y, z, 0)

	def curtain(self, state):
		_ipcSend(_VRUT_SWITCHCHILD, self.id-1, state, 'nil', 0, 0, 0, 0)
		
	def texture(self, textureObject, subObjectName):
		
		if isinstance(textureObject,PlayMediaObj):
			if textureObject.video:
				mediaobj = textureObject
				textureObject = addtexture(textureObject.playmediaObj)
				mediaobj.playmediaObj.command(1,mediaobj.filename)
				
				
			
		if type(textureObject) == types.InstanceType:
			# We have a texture object
			_ipcSend(_VRUT_APPLY_TEXTURE, textureObject.id-1, self.id-1, subObjectName, 0, 0, 0, 0)
		else:
			# We have a special command
			_ipcSend(_VRUT_APPLYMIRROR, self.id-1, 0, subObjectName, 0, 0, 0, 0)

	def center(self, x, y, z):
		_ipcSend(_VRUT_CENTERCHILD, self.id-1, 0, 'nil', x, y, z, 0)

	def scale(self, x, y, z):
		_ipcSend(_VRUT_SCALECHILD, self.id-1, 0, 'nil', x, y, z, 0)

	def movie(self, movieName, objectName, rate = 15.0, duration = -1.0, loopMode = 1, transp = 1):
		# example:  movie('astronaut', rate, duration, loopmode, target, transp)
		# print 'movie accessed'
		_ipcSend(_VRUT_MOVIE1CHILD, 0, 0, movieName, 0, 0, 0, 0)
		_ipcSend(_VRUT_MOVIE2CHILD, loopMode, transp, objectName, rate, duration, 0, 0)

	def appearance(self, flags = 0):
		_ipcSend(_VRUT_SETAPPEARANCE, self.id-1, flags, 'nil', 0, 0, 0, 0)

	def addchild(self, filename, anchor = WORLD):
		# in this case, WORLD is just a place holder because child get attached to parent
		# scene is also not used here because it'll get added to whatever 
		# scene the parent is in.
		filename.lower()
		return addchild(filename, anchor, 0, self.id-1)

	def sensor(self, sensorObject):
		if sensorObject.info == 'SENSOR':
			_ipcSend(_VRUT_ATTACHSENSOR, self.id-1, sensorObject.id, 'nil', 0, 0, 0, 0)
		else:
			_ipcSend(_VRUT_ATTACHRENDERER, self.id-1, sensorObject.id, 'nil', 0, 0, 0, 0)

	def link(self, object):
		if object.info == 'SENSOR':
			_ipcSend(_VRUT_ATTACHSENSOR, self.id-1, object.id, 'nil', 0, 0, 0, 0)
		else:
			_ipcSend(_VRUT_ATTACHRENDERER, self.id-1, object.id, 'nil', 0, 0, 0, 0)

	def unlink(self):
		_ipcSend(_VRUT_REMOVERENDERER, self.id-1, 0, 'nil', 0, 0, 0, 0)
	
	def alpha(self, level = -1):
		_ipcSend(_VRUT_ALPHACHILD, self.id-1, 0, 'nil', level, 0, 0, 0)

	def stats(self):
		_ipcSend(_VRUT_STATSCHILD, self.id-1, 0, 'nil', 0, 0, 0, 0)
	
	def getchild(self, subObjectName):
		global _numChild

		child = VRUTChild()

		child.id = _ipcSendHold(_VRUT_GETCHILD, self.id-1, 0, subObjectName, 0, 0, 0, 0)

		if child.id > 0:
			return child
		elif child.id == 0:
			print '** Subchild not found: ', subObjectName
		elif child.id == -1:
			_numChild = _numChild + 1
			child.id = _numChild
			return child

	def collision(self, action = START):
		_ipcSend(_VRUT_COLLISIONEYEPOINT, self.id-1, action, '', 0, 0, 0, 0)

	def script(self, scriptName):
		_ipcSend(_VRUT_SCRIPTCHILD, self.id-1, 0, scriptName, 0, 0, 0, 0)
		execute('import ' + scriptName)


	def callback(self, eventType, functionName):
		try:
			raise None
		except:
			c = sys.exc_traceback.tb_frame.f_back.f_code

		s = string.split(c.co_filename, '\\')
		s = string.split(s[1], '.')
		scriptName = s[0] + '.'

		_ipcSend(_VRUT_SELFCALLBACK, eventType, 0, scriptName + functionName, 0, 0, 0, 0)


	def starttimer(self, timerNum=1, timeSec=0.001):
		_ipcSend(_VRUT_SELFSTARTTIMER, timerNum, self.id-1, 'nil', timeSec, 0, 0, 0)


	def copy(self):
		global _numChild

		child = VRUTChild()

		child.id = _ipcSendHold(_VRUT_SELFCOPY, self.id-1, 0, 'nil', 0, 0, 0, 0)

		if child.id > 0:
			return child
		elif child.id == 0:
			print 'Could not make a copy of that child!'
		elif child.id == -1:
			_numChild = _numChild + 1
			child.id = _numChild
			return child


	def clone(self):
		global _numChild

		child = VRUTChild()

		child.id = _ipcSendHold(_VRUT_SELFCLONE, self.id-1, 0, 'nil', 0, 0, 0, 0)

		if child.id > 0:
			return child
		elif child.id == 0:
			print 'Could not make a clone of that child!'
		elif child.id == -1:
			_numChild = _numChild + 1
			child.id = _numChild
			return child


	def playsound(self, fileName = "", flag = 0):
		_ipcSend(_VRUT_SOUNDCHILD,  self.id-1, flag, fileName, 0, 0, 0, 0) 

	def color(self, red = 1, green = 1, blue = 1):
		_ipcSend(_VRUT_COLORTEXT, self.id-1, 0, 'nil', red, green, blue, 0)

	def width(self, width = 1.0):
		_ipcSend(_VRUT_WIDTHTEXT, self.id-1, 0, 'nil', width, 0, 0, 0)

	def message(self, message):
		if len(message) > 80:
			print '**ERROR: Message length cannot exceed 80 characters. (sent ', len(message),')'
		else:		
			_ipcSend(_VRUT_MESSAGETEXT, self.id-1, 0, message, 0, 0, 0, 0)

	def visible(self, state):
		if state == 1:
			_ipcSend(_VRUT_SWITCHCHILD, self.id-1, OPEN, 'nil', 0, 0, 0, 0)
		else:
			_ipcSend(_VRUT_SWITCHCHILD, self.id-1, CLOSE, 'nil', 0, 0, 0, 0)
	
	def update(self, transform):
		data = transform.get()

		_ipcSend(_VRUT_SETMATRIX1, self.id-1, _MATRIX_CHILD, 'nil', data[0][0], data[0][1], data[0][2], data[0][3])
		_ipcSend(_VRUT_SETMATRIX2, self.id-1, _MATRIX_CHILD, 'nil', data[1][0], data[1][1], data[1][2], data[1][3])
		_ipcSend(_VRUT_SETMATRIX3, self.id-1, _MATRIX_CHILD, 'nil', data[2][0], data[2][1], data[2][2], data[2][3])
		_ipcSend(_VRUT_SETMATRIX4, self.id-1, _MATRIX_CHILD, 'nil', data[3][0], data[3][1], data[3][2], data[3][3])
					
	
					
					
					
					
def addOSG(filename):
	
	num = _ipcSendHold(_VRUT_ADDOSGCHILD, 0, 0, filename, 0, 0, 0, 0)
	if num == -1:
		return
		
	child = OSGChild()
	child.id = num
	return child
	
#************************************************************
def addchild(fileName, anchor = WORLD, scene = 1, parent = -1):
	global _numChild

	#print ipc.getalive()
	child = VRUTChild()
	# must send -1 for param2 to signifiy that a child is an orphan
	child.id = _ipcSendHold(_VRUT_ADDCHILD, anchor, parent, fileName, scene, 0, 0, 0)

	# If the returned ID is positive, then everything worked fine.  If
	# ID equals 0 then the file wasn't found.  If ID equals -1, then the
	# command was blocked since winvrut wasn't running yet (this is part of
	# implementing the new method for doing callbacks).

	if child.id > 0:
		return child
	elif child.id == 0:
		print '** File not found: ', fileName
	elif child.id == -1:
		_numChild = _numChild + 1
		child.id = _numChild
		return child

	

#************************************************************
def addtext(message, anchor = WORLD, scene = 1, parent = -1):
	global _numChild

	if len(message) > 80:
		print '**ERROR: Message length cannot exceed 80 characters. (sent ', len(message),')'
		return

	#print ipc.getalive()
	child = VRUTChild()
	# must send -1 for param2 to signifiy that a child is an orphan
	child.id = _ipcSendHold(_VRUT_ADDTEXT, anchor, parent, message, scene, 0, 0, 0)

	# If the returned ID is positive, then everything worked fine.  If
	# ID equals 0 then the file wasn't found.  If ID equals -1, then the
	# command was blocked since winvrut wasn't running yet (this is part of
	# implementing the new method for doing callbacks).

	if child.id > 0:
		return child
	elif child.id == 0:
		print '**ERROR: File not found: ', fileName
	elif child.id == -1:
		_numChild = _numChild + 1
		child.id = _numChild
		return child



#************************************************************
def playsound(fileName = "", flag = 0):

	_ipcSend(_VRUT_PLAYSOUND,  flag, 0, fileName, 0, 0, 0, 0)



#************************************************************
class VRUTSensor:
	id = -1
	info = 'SENSOR'

	def reset(self):
		_ipcSend(_VRUT_RESETSENSOR, self.id, 0, 'nil', 0, 0, 0, 0)

	def get(self):
		return ipc.receive(-self.id)

	def center(self, x, y, z):
		_ipcSend(_VRUT_CENTERSENSOR, self.id, 0, 'nil', x, y, z, 0)

	def scale(self, x, y, z):
		_ipcSend(_VRUT_SCALESENSOR, self.id, 0, 'nil', x, y, z, 0)

	def command(self, issuedCommand, message = '', a=0, b=0, c=0):
		_ipcSend(_VRUT_COMMANDSENSOR, self.id, 0, message, issuedCommand, a, b, c)


	def starthotspot(self, hotspotNum, kind, xloc, zloc,  p1, p2 = 0, p3 = 0, yloc=0):
		# Negate Z center value so postitive Z is North bound
		list = xloc, -zloc, p1, p2, p3, yloc
		mesg = repr(list)

		if kind not in HOTSPOT_LIST:
			print '** Invalid hotspot kind.'
			return

		# -1 means that this hotspot's source is the eye position
		_ipcSend(_VRUT_STARTHOTSPOT, hotspotNum, kind, mesg, self.id, 0, 0, 0)

	def type(self, newType):
		if newType == HEAD_ORI:
			newType = HEAD_ORI + 64    # 64 for Euler 2 type
		_ipcSend(_VRUT_TYPESENSOR, self.id, newType, 'nil', 0, 0, 0, 0)


#************************************************************
def addsensor(sensorType):

	global _numSensor

	sensor = VRUTSensor()

	if type(sensorType) == types.StringType:
		sensor.id = _ipcSendHold(_VRUT_ADDSENSOR, 0, 0, sensorType, 0, 0, 0, 0)
	else:
		sensor.id = _ipcSendHold(_VRUT_ADDSENSOR, sensorType, 0, '', 0, 0, 0)


	if _where == 'X' and sensor.id >= 0:
		return sensor
	elif _where == 'X' and sensor.id == -1:
		print '** Sensor not found: ', sensorType
	elif _where == 'INT':
		_numSensor += 1
		sensor.id = _numSensor
		return sensor



#************************************************************
def version():

	_ipcSend(_VRUT_VERSION, 0, 0, vrut_version, ipc.getversion(), 0, 0, 0)


#************************************************************
def setaspect(horz2vertRatio):

	_ipcSend(_VRUT_SETASPECT, 1, 0, 'nil', horz2vertRatio, 0, 0, 0)


#************************************************************
#def mouseMotionOff():
#	_ipcSend(_VRUT_MOUSEMOTIONOFF, 0, 0, 'nil', 0, 0, 0, 0)
	

#************************************************************
#def cursorVisible(onOff = 1):
#	_ipcSend(_VRUT_CURSORVISIBLE, onOff, 0, 'nil', 0, 0, 0, 0)

#************************************************************
#def mouseScroll(state = START):
#	if state == START:
#		_ipcSend(_VRUT_MOUSESCROLL, 0, 0, 'nil', 0, 0, 0, 0)
#	elif state == STOP:
#		_ipcSend(_VRUT_MOUSEMOTIONOFF, 0, 0, 'nil', 0, 0, 0, 0)

#************************************************************
#def mouseConstrain(state = START):
#	if state == START:
#		_ipcSend(_VRUT_MOUSECONSTRAIN, 1, 0, 'nil', 0, 0, 0, 0)
#	elif state == STOP:
#		_ipcSend(_VRUT_MOUSECONSTRAIN, 0, 0, 'nil', 0, 0, 0, 0)

#************************************************************
def mouseData(stateX, stateY):

   if stateX == RELATIVE:
      _ipcSend(_VRUT_HORZ_RELATIVE, 0, 0, 'nil', 0, 0, 0, 0)
   elif stateX == ABSOLUTE:
      _ipcSend(_VRUT_HORZ_ABSOLUTE, 0, 0, 'nil', 0, 0, 0, 0)
      
   if stateY == RELATIVE:
      _ipcSend(_VRUT_VERT_RELATIVE, 0, 0, 'nil', 0, 0, 0, 0)
   elif stateY == ABSOLUTE:
      _ipcSend(_VRUT_VERT_ABSOLUTE, 0, 0, 'nil', 0, 0, 0, 0)


#************************************************************
def mouse(state):

	if state == START:
		_ipcSend(_VRUT_MOUSESCROLL, 0, 0, 'nil', 0, 0, 0, 0)
	elif state == STOP:
		_ipcSend(_VRUT_MOUSEMOTIONOFF, 0, 0, 'nil', 0, 0, 0, 0)
	
		
#************************************************************
def cursor(state = -1):

	# If no argument is passed, then signal VRUT with -1 and 
	# it'll just change the current state to the opposite.
	_ipcSend(_VRUT_CURSORVISIBLE, state, 0, 'nil', 0, 0, 0, 0)


#************************************************************
class VRUTWindow:
	id = -1

	def setposition(self, horzPos = 0, vertPos = 0):
		_ipcSend(_VRUT_SUBWINDOW_POS, self.id-1, 0, 'nil', horzPos, vertPos, 0, 0)

	def setsize(self, width = 320, height = 240):
		_ipcSend(_VRUT_SUBDINDOW_SIZE, self.id-1, 0, 'nil', width, height, 0, 0)

	def setviewpoint(self, viewPoint = 0):
		if viewPoint == 0:
			_ipcSend(_VRUT_SUBWINDOW_VIEWPOINT, self.id-1, 0, 'nil', 0, 0, 0, 0)
		else:
			_ipcSend(_VRUT_SUBWINDOW_VIEWPOINT, self.id-1, viewPoint.id-1, 'nil', 0, 0, 0, 0)

	def visible(self, state):
		_ipcSend(_VRUT_SUBWINDOW_CURTAIN, self.id-1, state, 'ni', 0, 0, 0, 0)
		


#************************************************************
def addwindow():

	global _numWindow

	window = VRUTWindow()
	window.id = _ipcSendHold(_VRUT_ADD_SUBWINDOW, 0, 0, 'nil', 0, 0, 0, 0)

	if window.id >= 0:
		return window
	elif window.id == 0:
		print '** No more windows allowed!!'
	elif window.id == -1:
		_numWindow += 1
		window.id = _numWindow
		return window
		

#************************************************************
class VRUTViewpoint:
	id = -1

	def setfov(self, vertFOV, h2vRatio):
		_ipcSend(_VRUT_SETFOV, self.id-1, 0, 'nil', vertFOV, h2vRatio, 0, 0)

	def setclip(self, near, far):
		_ipcSend(_VRUT_CLIP, self.id-1, 0, 'nil', near, far, 0, 0)
	
	def translate(self, xPos, yPos, zPos):
		_ipcSend(_VRUT_SUBVIEWPOSITION, self.id-1, 0, 'nil', xPos, yPos, zPos, 0)

	def rotate(self, x, y, z, deg=''):
		if deg == '':
			_ipcSend(_VRUT_SUBVIEWROTATEEULER, self.id-1, 0, 'nil', x, y, z, 0)
		else:
			_ipcSend(_VRUT_SUBVIEWROTATE, self.id-1, 0, 'nil', x, y, z, deg)
		

	def update(self, transform):

		data = transform.get()

		_ipcSend(_VRUT_SETMATRIX1, self.id-1, _MATRIX_VIEWPOINT, 'nil', data[0][0], data[0][1], data[0][2], data[0][3])
		_ipcSend(_VRUT_SETMATRIX2, self.id-1, _MATRIX_VIEWPOINT, 'nil', data[1][0], data[1][1], data[1][2], data[1][3])
		_ipcSend(_VRUT_SETMATRIX3, self.id-1, _MATRIX_VIEWPOINT, 'nil', data[2][0], data[2][1], data[2][2], data[2][3])
		_ipcSend(_VRUT_SETMATRIX4, self.id-1, _MATRIX_VIEWPOINT, 'nil', data[3][0], data[3][1], data[3][2], data[3][3])



#************************************************************
def addviewpoint():

	global _numViewpoint

	viewpoint = VRUTViewpoint()
	viewpoint.id = _ipcSendHold(_VRUT_ADD_SUBVIEWPOINT, 0, 0, 'nil', 0, 0, 0, 0)

	if viewpoint.id >= 0:
		return viewpoint
	elif viewpoint.id == 0:
		print '** No more viewpoints allowed!!'
	elif viewpoint.id == -1:
		_numViewpoint += 1
		viewpoint.id = _numViewpoint
		return viewpoint

	


#************************************************************
def override():
	# eventually this command will be used to push override states onto the
	# override stack, but right now when called without arguments it'll 
	# be used to pop the stack.

	_ipcSend(_VRUT_OVERRIDE, 0, 0, 'nil', 0, 0, 0, 0)



#************************************************************
class VRUTMail:
	id = -1
	
	def send(self, textMessage, mailCode = 0):
		_ipcSend(_VRUT_SENDMAIL, mailCode, self.id-1, textMessage, 0, 0, 0, 0)
		
		
#************************************************************
def addmail(mailCode = ''):
	
	global _numMail

	mailbox = VRUTMail()

	if type(mailCode) == types.StringType:
		if len(mailCode) > 35:
			print '***Name too long (must be less than 35 characters)'
			return
		else:
			mailbox.id = _ipcSendHold(_VRUT_ADDMAIL, 0, 0, mailCode, 0, 0, 0, 0)

	elif type(mailCode) == types.IntType:
		_ipcSend(_VRUT_SELFADDMAIL, mailCode, 0, 'nil', 0, 0, 0, 0)

	else:
		print '***Invalid mail code type!'
		return
	
	#mailbox.id = 1
	
	if mailbox.id == -1:
		_numMail = _numMail + 1
		mailbox.id = _numMail
		return mailbox
	else:
		return mailbox
	

#************************************************************
def scene(sceneNum, eye = LEFT_EYE | RIGHT_EYE):

	_ipcSend(_VRUT_SETSCENE, sceneNum, eye, 'nil', 0, 0, 0, 0)
	
def setscene(sceneNum, eye = LEFT_EYE | RIGHT_EYE):

	scene(sceneNum, eye)

	
	
#************************************************************
def screencapture(fileName = 'vrutcapture.jpg'):

	_ipcSend(_VRUT_SCREENCAPTURE, 0, 0, fileName, 0, 0, 0, 0)
	
	

#************************************************************
def savecsb(fileName):

	_ipcSend(_VRUT_SAVECSB, 0, 0, fileName, 0, 0, 0, 0)



#************************************************************
def findchild(objectName):

	_ipcSend(_VRUT_FINDCHILD, 0, 0, objectName, 0, 0, 0, 0)


#************************************************************
def splashscreen(textureName):

	_ipcSend(_VRUT_SPLASHSCREEN, 0, 0, textureName, 0, 0, 0, 0)



#************************************************************
def self():

	child = VRUTChild()

	child.id = _ipcSendHold(_VRUT_SELF, 0, 0, '', 0, 0, 0, 0)

	return child




#************************************************************
def interactive():

	global _ipcSend, _ipcSendHold
	
	_ipcSend = ipc.send
	_ipcSendHold = ipc.sendhold

	#print 'Interactive mode'
	

#************************************************************
def waitkeypress():
	if _where == 'X':
		return ipc.getkeyboard()
	else:
		return -1


#************************************************************
def stayontop():
	_ipcSend(_VRUT_STAYONTOP, 0, 0, '', 0, 0, 0, 0)



#************************************************************
def settrackingtype(sensorType):
	_ipcSend(_VRUT_SETTRACKING_TYPE, sensorType, 0, 'nil', 0, 0, 0, 0)


#************************************************************
def setinitok():
	_ipcSend(_VRUT_SETINITOK, 0, 0, 'nil', 0, 0, 0, 0)


#************************************************************
def noprojection():
	_ipcSend(_VRUT_NOPROJECTION, 0, 0, 'nil', 0, 0, 0, 0)


#************************************************************
class VRUTRenderer:
	id = -1
	info = 'RENDERER'

	def reset(self):
		_ipcSend(_VRUT_RESETRENDERER, self.id, 0, 'nil', 0, 0, 0, 0)

	def command(self, issuedCommand, message = '', a=0, b=0, c=0):
		_ipcSend(_VRUT_COMMANDRENDERER, self.id, 0, message, issuedCommand, a, b, c)



#************************************************************
def addrenderer(rendererType):

	global _numRenderer

	renderer = VRUTRenderer()

	if type(rendererType) == types.StringType:
		renderer.id = _ipcSendHold(_VRUT_ADDRENDERER, 0, 0, rendererType, 0, 0, 0, 0)
	else:
		renderer.id = _ipcSendHold(_VRUT_ADDRENDERER, rendererType, 0, '', 0, 0, 0)


	if _where == 'X' and renderer.id >= 0:
		return renderer
	elif _where == 'X' and renderer.id == -1:
		print '** Renderer not found: ', rendererType
	elif _where == 'INT':
		_numRenderer += 1
		renderer.id = _numRenderer
		return renderer




#************************************************************
class Light:

	id = -1

	def ambient(self, red, green, blue):
		lightambient(red, green, blue)

	def color(self, red, green, blue):
		lightcolor(self.id+1, red, green, blue)

	def direction(self, vectorX, vectorY, vectorZ):
		lightdirection(self.id+1, vectorX, vectorY, vectorZ)

	def disable(self):
		lightdisable(self.id+1)

	def enable(self):
		lightenable(WORLD, self.id+1)

	def intensity(self, intensity):
		lightintensity(self.id+1, intensity)

	def position(self, posX, posY, posZ):
		lightposition(self.id+1, posX, posY, posZ)

	def spread(self, angle):
		lightspread(self.id+1, angle)

	def spotexponent(self, exponent):
		_ipcSend(_VRUT_LIGHTSPOTEXPONENT, self.id, exponent, 'nil', 0, 0, 0, 0)


#************************************************************
def addlight():
	global _numLights

	if _numLights == 8:
		print "** ERROR: Can't have more than 8 lights"
		return

	light = Light()
	light.id = _numLights

	_numLights += 1

	return light


#************************************************************
#************************************************************
#************************************************************
#************************************************************


##mytimer = viz.add(viz.TIMER_EVENT, handleTimer)
##mytimer.alarm(id, delay, numRepeat)
##
##mykeys = viz.add(viz.KEYBOARD_EVENT, handleKeys)
##
##mynetwork = viz.add(viz.NETWORK_EVENT, handleNet)
##mynetwork.send(toWho, what)
##

##class NewEvent:
##
##	count = [0]
##	funcPointers = []
##
##
##	def __init__(self, callBackFunction):
##		self.id = self.count[0]
##		self.count[0] = self.count[0] + 1
##		self.type = -1
##		self.funcPointers.append(function)
##
##	def start(self):
##		0
##
##	def stop(self):
##		0
##
##	def callback(self, function):
##		self.funcPointers[self.id] = function
##		_ipcSend(REGISTER_CALLBACK, self.type, self.id, '', 0, 0, 0, 0)
##		
##
##	def runfunction(self, id, p1, p2=0, p3=0):
##		pass
##
##	def status(self):
##		print self.count, self.funcPointers
##		
##
###************************************************************
##class EventTimer(NewEvent):
##
##	def alarm(self, ID, delay, repeats):
##		pass
##	def kill():
##		pass
##		


#************************************************************
def _HandleGenericEvent(type, data1, d2='', d3='', d4=''):

	if type == TIMER_EVENT:
		_EventFunctionPointers[TIMER_EVENT](data1)

	elif type == KEYBOARD_EVENT:
		_EventFunctionPointers[KEYBOARD_EVENT](data1)

	elif type == MAIL_EVENT:
		_EventFunctionPointers[MAIL_EVENT](data1)

	elif type == MOUSEMOVE_EVENT:
		_EventFunctionPointers[MOUSEMOVE_EVENT](data1, d2)

	elif type == MOUSEBUTTON_EVENT:
		_EventFunctionPointers[MOUSEBUTTON_EVENT](data1)

	elif type == HOTSPOT_EVENT:
		_EventFunctionPointers[HOTSPOT_EVENT](data1, d2, d3, d4)

	elif type == COLLISION_EVENT:
		_EventFunctionPointers[COLLISION_EVENT](data1)

	elif type == KEYBOARDX_EVENT:
		_EventFunctionPointers[KEYBOARDX_EVENT](data1)



#************************************************************
def addmultimedia(fileName, video):

	p = 0
	
	if video:
		p = addsensor('movietex')
	else:
		p = addsensor('playaudio')

	if p:
		pObj = PlayMediaObj()
		pObj.playmediaObj = p
		pObj.video = video
		if not video:
			p.command(1, fileName)
		else:
			pObj.filename = fileName
		return pObj




TypeBitmaps = ['JPG', 'BMP', 'TIF', 'GIF', 'PNG', 'RGB', 'RGBA']
TypeAudio = ['WAV', 'MP3', 'MID', 'MIDI']
TypeVideo = ['AVI', 'MPG', 'MPEG']
TypeOSG = ['OSG','3DS']
#************************************************************
def add(what, param1='', param2='', param3=''):

	if type(what) == types.IntType:
		if what == WINDOW:
			return addwindow()
		elif what == VIEWPOINT:
			return addviewpoint()
		elif what == TEXT3D:
			if param2 != '':
				return addtext(param1, param2)
			else:
				return addtext(param1)
		elif what == MAILBOX:
			return addOSC(param1)
			
	else:

		s = what.split('.')
		fileType = s[len(s)-1].upper()

		#VRMLs
		if fileType == 'WRL':
			if param3 != '':
				return addchild(what, param1, param2, param3)
			elif param2 != '':
				return addchild(what, param1, param2)
			elif param1 != '':
				return addchild(what, param1)
			else:
				return addchild(what)
		
		#OSG Readable file
		elif fileType in TypeOSG:
			return addOSG(what)
			
		# Bitmaps
		elif fileType in TypeBitmaps:
			return addtexture(what)

		# Audio
		elif fileType in TypeAudio:
			return addmultimedia(what,0)
		
		# Video
		elif fileType in TypeVideo:
			return addmultimedia(what,1)

		# Sensor plug-in
		elif fileType == 'DLS':
			return addsensor(s[0])

		elif fileType == 'DLR':
			return addrenderer(s[0])


