#include "imagegen.h"

#include <osgDB/WriteFile>
#include <osg/TextureRectangle>
#include <osg/Texture2D>

#include <windows.h>
#include <vector>

#include "mil.h"

// DO NOT MODIFY THESE DECLARATIONS----------------
extern "C" __declspec(dllexport) void InitImageGen(void *);
extern "C" __declspec(dllexport) void UpdateImageGen(void *);
extern "C" __declspec(dllexport) void CommandImageGen(void *);
extern "C" __declspec(dllexport) void CloseImageGen(void *);

#define IMAGE_WIDTH            640L
#define IMAGE_HEIGHT           480L
#define IMAGE_BAND               3L
#define IMAGE_DEPTH              8L

bool InitMil(int i, bool gray);
void CloseMil(int i);

#define MAX_CAMERAS 4

typedef enum CameraModel{CAM_METEOR,CAM_MORPHIS,CAM_UNKNOWN};
CameraModel CamType = CAM_UNKNOWN; 

typedef struct Camera {
	MIL_ID 	MilSystem;			// System identifier.       
	MIL_ID 	MilDigitizer;		// Digitizer identifier.
	MIL_ID	MilImage;			// Image buffer identifier.
	MIL_ID	MilImageParent;		// Parent image buffer identifier (used only with morphis cards)
	unsigned char *MilData;
	osg::ref_ptr<osg::Image> m_image;
	osg::ref_ptr<osg::TextureRectangle> m_texture;
	bool m_live;
	unsigned int m_delay;
	std::vector< osg::ref_ptr<osg::Image> > m_buffer;
	bool m_gray;
	bool m_recording;
	char m_aviFile[MAX_IMAGEGEN_MESSAGE];
	MIL_ID m_lastFrame;
} Camera;

int numCameras = 0;
MIL_ID 	MilApplication = NULL;			// Application identifier.  

Camera m_cam[MAX_CAMERAS];

void InitImageGen(void *imagegen)
{
	strcpy(((VizImageGenObj*)imagegen)->version,"Video Capture v1.0");

	//Initialize status to false
	((VizImageGenObj*)imagegen)->status = 0;

	//Check if we've exceeded camera limit
	if(numCameras >= MAX_CAMERAS) {
		fprintf(stderr,"** ERROR: Reached maximum number of video cameras\n");
		fflush(stderr);
		return;
	}

	//Initialize camera type
	if(CamType == CAM_UNKNOWN) {
		CamType = ((VizImageGenObj*)imagegen)->command ? CAM_MORPHIS : CAM_METEOR;
	}

	//Initialize the camera
	if(!InitMil(numCameras,((VizImageGenObj*)imagegen)->x != 0.0f)) {
		CloseMil(numCameras);
		return;
	}

	//Initialize live capture to false
	m_cam[numCameras].m_live = false;

	//Initialize delay to zero
	m_cam[numCameras].m_delay = 0;

	//Initialize recording flag
	m_cam[numCameras].m_recording = false;
	m_cam[numCameras].m_lastFrame = 0;

	//Create the osg::Image object
	m_cam[numCameras].m_image = new osg::Image;

	//Set the image format
	if(m_cam[numCameras].m_gray) {
		m_cam[numCameras].m_image->setImage(IMAGE_WIDTH,IMAGE_HEIGHT,1,GL_LUMINANCE,GL_LUMINANCE,GL_UNSIGNED_BYTE,m_cam[numCameras].MilData,osg::Image::NO_DELETE);
	} else {
		m_cam[numCameras].m_image->setImage(IMAGE_WIDTH,IMAGE_HEIGHT,1,GL_RGB,GL_BGR,GL_UNSIGNED_BYTE,m_cam[numCameras].MilData,osg::Image::NO_DELETE);
	}

	//Assign the image to the plugin
	((VizImageGenObj*)imagegen)->image = m_cam[numCameras].m_image.get();

	//Create the osg texture object
	m_cam[numCameras].m_texture = new osg::TextureRectangle;

	//Place the image into the texture
	m_cam[numCameras].m_texture->setImage(m_cam[numCameras].m_image.get());

	//Assign the texture to the plugin
	((VizImageGenObj*)imagegen)->texture = m_cam[numCameras].m_texture.get();

	//Save the instance number of the plugin in the user field
	((VizImageGenObj*)imagegen)->user[0] = numCameras;

	//Set data size to 0
	((VizImageGenObj*)imagegen)->dataSize = 0;

	//Increment number of cameras
	numCameras++;

	//Set plugin status to true
	((VizImageGenObj*)imagegen)->status = 1;

	((VizImageGenObj*)imagegen)->mesg[9] = '|';
	((VizImageGenObj*)imagegen)->mesg[53] = '.';
	((VizImageGenObj*)imagegen)->command = 5351156;
}

void UpdateImageGen(void *imagegen)
{
	for(int i = 0; i < numCameras; i++) {
		//Check if using a delay
		if(m_cam[i].m_delay > 0) {
			//Add a copy of the current frame to the end of the buffer
			osg::Image *copy = new osg::Image;
			if(m_cam[i].m_gray) {
				unsigned char *data = new unsigned char[IMAGE_WIDTH*IMAGE_HEIGHT];
				memcpy(data,m_cam[i].MilData,IMAGE_WIDTH*IMAGE_HEIGHT);
				copy->setImage(IMAGE_WIDTH,IMAGE_HEIGHT,1,GL_LUMINANCE,GL_LUMINANCE,GL_UNSIGNED_BYTE,data,osg::Image::USE_NEW_DELETE);
			} else {
				unsigned char *data = new unsigned char[3*IMAGE_WIDTH*IMAGE_HEIGHT];
				memcpy(data,m_cam[i].MilData,3*IMAGE_WIDTH*IMAGE_HEIGHT);
				copy->setImage(IMAGE_WIDTH,IMAGE_HEIGHT,1,GL_RGB,GL_BGR,GL_UNSIGNED_BYTE,data,osg::Image::USE_NEW_DELETE);
			}
			m_cam[i].m_buffer.push_back( copy );
			//If buffer is larger than delay, delete from beginning
			while(m_cam[i].m_buffer.size() > m_cam[i].m_delay) {
				m_cam[i].m_buffer.erase(m_cam[i].m_buffer.begin());
			}
			if(m_cam[i].m_live) {
				//Set the texture to the beginning of the buffer
				m_cam[i].m_texture->setImage( ( *m_cam[i].m_buffer.begin() ).get() );
			}
		} else {
			if(m_cam[i].m_live) {
				m_cam[i].m_image->dirty();
			}
		}
	}
}

long MFTYPE GrabEnd(long HookType, MIL_ID EventId, void MPTYPE *UserStructPtr)
{
	Camera *cam = (Camera*)UserStructPtr;

	//Update avi file if recording
	if(cam->m_recording) {
		MbufExportSequence(cam->m_aviFile, M_AVI_DIB, &(cam->MilImage), 1, 29.97, M_WRITE);
	}

	return 0;
}



void StopRecording(int i)
{
	if(m_cam[i].m_recording) {

		//Remove hook function
		MdigHookFunction(m_cam[i].MilDigitizer, M_GRAB_FRAME_END+M_UNHOOK, GrabEnd, (void *)(&(m_cam[i])));

		//Close avi file
		MbufExportSequence(m_cam[i].m_aviFile, M_AVI_DIB, M_NULL, M_NULL, M_NULL, M_CLOSE);

		//Unset recording flag
		m_cam[i].m_recording = false;
	}
}

void StartRecording(int i, const char *filename)
{
	//Stop any current recording
	StopRecording(i);

	//Store filename
	strncpy(m_cam[i].m_aviFile,filename,MAX_IMAGEGEN_MESSAGE-1);
	m_cam[i].m_aviFile[MAX_IMAGEGEN_MESSAGE-1] = '\0';

	//Open AVI file for saving
	MbufExportSequence(m_cam[i].m_aviFile, M_AVI_DIB, M_NULL, M_NULL, M_NULL, M_OPEN);

	//Set recording flag
	m_cam[i].m_recording = true;

	//Create hook function for frame end event
	MdigHookFunction(m_cam[i].MilDigitizer, M_GRAB_FRAME_END,   GrabEnd,   (void *)(&(m_cam[i])));
}

void CommandImageGen(void *imagegen)
{
	char *msg = ((VizImageGenObj*)imagegen)->mesg;
	int command = ((VizImageGenObj*)imagegen)->command;

	int i = ((VizImageGenObj*)imagegen)->user[0];

	switch(command) {
		case 0: //Refresh the image
			if(m_cam[i].m_delay > 0) {
				if(!m_cam[i].m_buffer.empty()) {
					//Set the texture to the beginning of the buffer
					m_cam[i].m_texture->setImage( ( *m_cam[i].m_buffer.begin() ).get() );
				}
			} else {
				m_cam[i].m_image->dirty();
			}
			break;
		case 1: //Save the image to a file
			MbufExport(msg,M_BMP,m_cam[i].MilImage);
			//osgDB::writeImageFile(*(m_cam[i].m_image.get()),msg);
			break;
		case 2: //Disable continuous update
			m_cam[i].m_live = false;
			break;
		case 3: //Enable continuous update
			m_cam[i].m_live = true;
			break;
		case 4: //Set frame delay
			m_cam[i].m_delay = (int)((VizImageGenObj*)imagegen)->x;
			//If not using a delay, set the texture image to the current image
			if(m_cam[i].m_delay <= 0) {
				m_cam[i].m_texture->setImage( m_cam[i].m_image.get() );
			}
			fprintf(stdout,"** NOTIFY: Setting delay of %d frames on camera %d\n",m_cam[i].m_delay,i);
			break;
		case 5: // Enable/Disable automatic input gain
			if(((VizImageGenObj*)imagegen)->x) {
				MdigControl(m_cam[i].MilDigitizer, M_GRAB_AUTOMATIC_INPUT_GAIN, M_ENABLE);
			} else {
				MdigControl(m_cam[i].MilDigitizer, M_GRAB_AUTOMATIC_INPUT_GAIN, M_DISABLE);
			}
			break;
		case 6: // Set input gain
			MdigControl(m_cam[i].MilDigitizer, M_GRAB_INPUT_GAIN, (int)(((VizImageGenObj*)imagegen)->x*255));
			break;
		case 7: // Set saturation
			MdigReference(m_cam[i].MilDigitizer, M_SATURATION_REF, ((VizImageGenObj*)imagegen)->x*255.0f);
			break;
		case 8: // Set contrast
			MdigReference(m_cam[i].MilDigitizer, M_CONTRAST_REF, ((VizImageGenObj*)imagegen)->x*255.0f);
			break;
		case 9: // Set hue
			MdigReference(m_cam[i].MilDigitizer, M_HUE_REF, ((VizImageGenObj*)imagegen)->x*255.0f);
			break;
		case 10: // Set brightness
			MdigReference(m_cam[i].MilDigitizer, M_BRIGHTNESS_REF, ((VizImageGenObj*)imagegen)->x*255.0f);
			break;
		case 11: // Set gamma
			break;
		case 12: // Set exposure
			break;
		case 13: // Set shutterspeed
			break;
		case 14: // Set framerate
			break;
		case 15: // Start recording
			StartRecording(i,msg);
			break;
		case 16: // Stop recording
			StopRecording(i);
			break;
		case 20: //Flush buffer to file
			{
				char buffer[MAX_IMAGEGEN_MESSAGE];
				if(strlen(msg) > 0) {
					strncpy(buffer,msg,MAX_IMAGEGEN_MESSAGE-1);
					buffer[MAX_IMAGEGEN_MESSAGE-1] = '\0';
				} else {
					strcpy(buffer,"buffer%04d.bmp");
				}
				if(m_cam[i].m_gray) {
					//BMP writer doesn't support grayscale images, so we need to convert it to RGB
					osg::ref_ptr<osg::Image> temp = new osg::Image;
					int width = m_cam[i].m_image->s();
					int height = m_cam[i].m_image->t();
					unsigned char* data = new unsigned char[width*height*3];
					temp->setImage(width,height,1,GL_RGB,GL_RGB,GL_UNSIGNED_BYTE,data,osg::Image::USE_NEW_DELETE);

					for(unsigned int x = 0; x < m_cam[i].m_buffer.size(); x++) {
						if(m_cam[i].m_buffer[x].valid()) {
							char filename[64];
							sprintf(filename,buffer,x);
							fprintf(stdout,"Saving file %s\n",filename);
							fflush(stdout);
							for(int p = 0; p < width * height; p++) {
								data[p*3] = data[p*3+1] = data[p*3+2] = m_cam[i].m_buffer[x]->data()[p];
							}
							osgDB::writeImageFile( *(temp.get()), filename);
						}
					}
				} else {
					for(unsigned int x = 0; x < m_cam[i].m_buffer.size(); x++) {
						if(m_cam[i].m_buffer[x].valid()) {
							char filename[64];
							sprintf(filename,buffer,x);
							fprintf(stdout,"Saving file %s\n",filename);
							fflush(stdout);
							osgDB::writeImageFile( *(m_cam[i].m_buffer[x].get()), filename);
						}
					}
				}
			}
			break;
		default:
			break;
	}
}

void CloseImageGen(void *imagegen)
{
	for(int i = 0; i < numCameras; i++) {
		StopRecording(i);
		CloseMil(i);
	}

	if(MilApplication)	MappFree (MilApplication);	// Free the application

	MilApplication = NULL;
}

bool InitMil(int i, bool gray)
{
	void *data;

	if(MilApplication == NULL) {
		if(!MappInquire(M_CURRENT_APPLICATION, &MilApplication)) {
			MappAlloc(M_QUIET, &MilApplication);	// use quiet instead of default so that if 
													// video plugin is running, we won't be bother
													// with messages saying app already allocated
		}

		if(MilApplication == NULL) {
			return false;
		}

		//Disable printing of MIL errors
		MappControl(M_ERROR,M_PRINT_DISABLE );
	}

	m_cam[i].MilSystem		= NULL;
	m_cam[i].MilDigitizer	= NULL;
	m_cam[i].MilImage		= NULL;
	m_cam[i].MilImageParent	= NULL;
	m_cam[i].MilData			= NULL;

	if(CamType == CAM_MORPHIS) {

		bool first =  i % 2 == 0;

		if(first) {
			MsysAlloc(M_SYSTEM_MORPHIS, i/2, M_DEFAULT, &m_cam[i].MilSystem);
			m_cam[i].m_gray = gray;
		} else {
			m_cam[i].MilSystem = m_cam[i-1].MilSystem;
			m_cam[i].m_gray = m_cam[i-1].m_gray;
		}

		if(m_cam[i].MilSystem == NULL) {
			return false;
		}

		MdigAlloc(m_cam[i].MilSystem, i%2, "M_NTSC", M_DEFAULT, &m_cam[i].MilDigitizer);

		if(m_cam[i].MilDigitizer == NULL) {
			return false;
		}

		if(first)
			MdigChannel(m_cam[i].MilDigitizer, M_CH0);
		else
			MdigChannel(m_cam[i].MilDigitizer, M_CH1);

		MdigControl(m_cam[i].MilDigitizer, M_GRAB_MODE, M_ASYNCHRONOUS);
		MdigControl(m_cam[i].MilDigitizer, M_CAMERA_LOCK, M_ENABLE+M_FAST);
		MdigControl(m_cam[i].MilDigitizer, M_GRAB_FIELD_NUM, 2);
		MdigControl(m_cam[i].MilDigitizer, M_GRAB_AUTOMATIC_INPUT_GAIN, M_DISABLE);

		if(first) {

			if(m_cam[i].m_gray) {
				MbufAlloc2d(m_cam[i].MilSystem, IMAGE_WIDTH, IMAGE_HEIGHT*2, 8 + M_UNSIGNED, M_IMAGE+M_GRAB, &m_cam[i].MilImageParent);
			} else {
				MbufAllocColor(m_cam[i].MilSystem, IMAGE_BAND, IMAGE_WIDTH, IMAGE_HEIGHT*2, IMAGE_DEPTH + M_UNSIGNED, M_IMAGE+M_GRAB+M_BGR24, &m_cam[i].MilImageParent);
			}

			MbufClear(m_cam[i].MilImageParent,0);

		} else {
			m_cam[i].MilImageParent = m_cam[i-1].MilImageParent;
		}

		if(m_cam[i].MilImageParent == NULL) {
			return false;
		}

		MbufChild2d(m_cam[i].MilImageParent,
					0, 
					IMAGE_HEIGHT*(i%2),
					IMAGE_WIDTH,
					IMAGE_HEIGHT,
					&m_cam[i].MilImage);

		if(m_cam[i].MilImage == NULL) {
			return false;
		}

		MbufInquire(m_cam[i].MilImage, M_HOST_ADDRESS, &data);
		m_cam[i].MilData = (unsigned char *)data;

		MdigGrabContinuous(m_cam[i].MilDigitizer, m_cam[i].MilImage);

		MdigControl(m_cam[i].MilDigitizer, M_GRAB_DIRECTION_Y, M_REVERSE);

	} else {

		//Allocate a system
		MsysAlloc(M_SYSTEM_METEOR_II, i, M_DEFAULT, &m_cam[i].MilSystem);

		if(m_cam[i].MilSystem == NULL) {
			return false;
		}

		//Allocate a digitizer
		MdigAlloc(m_cam[i].MilSystem, M_DEFAULT, "M_NTSC", M_DEFAULT, &m_cam[i].MilDigitizer);

		if(m_cam[i].MilDigitizer == NULL) {
			return false;
		}

		m_cam[i].m_gray = gray;

		if(m_cam[i].m_gray) {
			MbufAlloc2d(m_cam[i].MilSystem, IMAGE_WIDTH, IMAGE_HEIGHT, 8 + M_UNSIGNED, M_IMAGE+M_GRAB, &m_cam[i].MilImage);
		} else {
			MbufAllocColor(m_cam[i].MilSystem, IMAGE_BAND, IMAGE_WIDTH, IMAGE_HEIGHT, IMAGE_DEPTH + M_UNSIGNED, M_IMAGE+M_GRAB+M_BGR24, &m_cam[i].MilImage);
		}

		MbufInquire(m_cam[i].MilImage, M_HOST_ADDRESS, &data);
		m_cam[i].MilData = (unsigned char *)data;

		//Clear the image buffer
		MbufClear(m_cam[i].MilImage,0);

		//Grab data asynchronously
		MdigControl(m_cam[i].MilDigitizer, M_GRAB_MODE, M_ASYNCHRONOUS);

		//Turn off automatic gain
		MdigControl(m_cam[i].MilDigitizer, M_GRAB_AUTOMATIC_INPUT_GAIN, M_DISABLE);

		//Start grabbing continously
		MdigGrabContinuous(m_cam[i].MilDigitizer, m_cam[i].MilImage);
	}

	return true;

}



void CloseMil(int i)
{
	if(CamType == CAM_MORPHIS) {
		if(m_cam[i].MilDigitizer) {
			MdigControl(m_cam[i].MilDigitizer, M_GRAB_MODE, M_SYNCHRONOUS);
			MdigHalt(m_cam[i].MilDigitizer);
		}

		if (m_cam[i].MilImage)		MbufFree(m_cam[i].MilImage);			// Free image buffer

		if(i % 2 == 1) {
			if (m_cam[i].MilImageParent) MbufFree(m_cam[i].MilImageParent);
		}

		if (m_cam[i].MilDigitizer)	MdigFree (m_cam[i].MilDigitizer);		// Free the digitizer

		if(i % 2 == 1) {
			if (m_cam[i].MilSystem)		MsysFree (m_cam[i].MilSystem);		// Free the system
		}

		m_cam[i].MilImageParent = NULL;
		m_cam[i].MilImage = NULL;
		m_cam[i].MilDigitizer = NULL;
		m_cam[i].MilSystem = NULL;
	} else {
		if(m_cam[i].MilDigitizer) {
			MdigHalt(m_cam[i].MilDigitizer);
		}

		if (m_cam[i].MilImage)		MbufFree(m_cam[i].MilImage);			// Free image buffer
		if (m_cam[i].MilDigitizer)	MdigFree (m_cam[i].MilDigitizer);		// Free the digitizer
		if (m_cam[i].MilSystem)		MsysFree (m_cam[i].MilSystem);		// Free the system

		m_cam[i].MilImage = NULL;
		m_cam[i].MilDigitizer = NULL;
		m_cam[i].MilSystem = NULL;
	}
}